/*
 * Top frame
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: TopFrameView.java 233 2008-10-08 18:07:06Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.KeyAdapter;
import java.awt.event.MouseAdapter;
import java.text.DateFormat;
import java.util.Date;
import java.util.logging.Level;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.BevelBorder;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;

/**
 * 最上位ビュー。
 * 最初に表示されるフレーム。
 * あらゆるコンポーネント、フレームの親。
 */
@SuppressWarnings("serial")
public class TopFrameView extends JFrame{

    private static final String INITCARD = "INITCARD";
    private static final String LANDCARD = "LANDINFO";
    private static final String BROWSECARD = "BROWSER";

    private static DateFormat dform =
            DateFormat.getDateTimeInstance(DateFormat.FULL,
                                           DateFormat.FULL);

    private final JComponent cards;
    private final CardLayout cardLayout = new CardLayout();

    private final LandsTree landsTreeView = new LandsTree();

    private final JPanel landInfo = new JPanel();

    private final JTextField sysMessage = new JTextField();
    private final JProgressBar progressBar = new JProgressBar();

    private final TabBrowser tabBrowser = new TabBrowser();

    private JComponent browsePanel;
    
    /**
     * トップフレームを生成する。
     */
    public TopFrameView(){
        super();
        setFrameTitle(null);

        this.cards = createCards();
        JComponent split = createSplitPane(this.landsTreeView, this.cards);
        JComponent statusBar = createStatusBar();

        Container content = getContentPane();
        BorderLayout layout = new BorderLayout();
        content.setLayout(layout);
        content.add(split, BorderLayout.CENTER);
        content.add(statusBar, BorderLayout.SOUTH);

        Component glassPane = new JComponent() {};
        glassPane.addMouseListener(new MouseAdapter() {});
        glassPane.addKeyListener(new KeyAdapter() {});
        setGlassPane(glassPane);
        
        return;
    }

    /**
     * カードパネルを生成する。
     * @return カードパネル
     */
    private JComponent createCards(){
        this.browsePanel = createBrowsePanel();
        
        JPanel panel = new JPanel();
        panel.setLayout(this.cardLayout);
        panel.add(INITCARD, createInitPanel());
        panel.add(LANDCARD, createLandInfo());
        panel.add(BROWSECARD, this.browsePanel);

        return panel;
    }

    /**
     * 初期パネルを生成
     * @return 初期パネル
     */
    private JComponent createInitPanel(){
        JLabel initMessage = new JLabel("村を選択してください");
        
        StringBuilder warn = new StringBuilder();
        warn.append("※ たまにはWebブラウザでアクセスして、");
        warn.append("<br></br>");
        warn.append("運営の動向を確かめようね！");
        warn = HTMLUtils.elemented("center",warn);
        warn = HTMLUtils.elemented("body", warn);
        warn = HTMLUtils.elemented("html", warn);
        JLabel warnMessage = new JLabel(warn.toString());
        
        JPanel panel = new JPanel();
        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();
        panel.setLayout(layout);
        
        constraints.weightx = 1.0;
        constraints.weighty = 1.0;
        constraints.anchor = GridBagConstraints.CENTER;
        constraints.gridx = GridBagConstraints.REMAINDER;
        panel.add(initMessage, constraints);
        panel.add(warnMessage, constraints);
        
        JScrollPane scrollPane = new JScrollPane(panel);

        return scrollPane;
    }

    /**
     * 国別情報を生成
     * @return 国別情報
     */
    private JComponent createLandInfo(){
        this.landInfo.setBorder(new EmptyBorder(5, 5, 5, 5));
        JScrollPane scrollPane = new JScrollPane(this.landInfo);

        return scrollPane;
    }

    /**
     * 内部ブラウザを生成
     * @return 内部ブラウザ
     */
    private JComponent createBrowsePanel(){
        JPanel panel = new JPanel();
        BorderLayout layout = new BorderLayout();
        panel.setLayout(layout);

        panel.add(this.tabBrowser, BorderLayout.CENTER);

        return panel;
    }

    /**
     * ブラウザ用ツールバーをセットする
     * @param toolbar ツールバー
     */
    public void setBrowseToolBar(JToolBar toolbar){
        this.browsePanel.add(toolbar, BorderLayout.NORTH);
        return;
    }
    
    /**
     * SplitPaneを生成
     * @param left 左コンポーネント
     * @param right 右コンポーネント
     * @return SplitPane
     */
    private JComponent createSplitPane(JComponent left, JComponent right){
        JSplitPane split = new JSplitPane();
        split.setLeftComponent(left);
        split.setRightComponent(right);
        split.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
        split.setContinuousLayout(false);
        split.setOneTouchExpandable(true);
        split.setDividerLocation(200);

        return split;
    }

    /**
     * ステータスバーを生成する。
     * @return ステータスバー
     */
    private JComponent createStatusBar(){
        this.sysMessage.setText(Jindolf.TITLE + " を使ってくれてありがとう！");
        this.sysMessage.setEditable(false);
        Border border = new BevelBorder(BevelBorder.LOWERED);
        border = new CompoundBorder(border, new EmptyBorder(2, 5, 2, 2));
        this.sysMessage.setBorder(border);

        this.progressBar.setIndeterminate(false);
        this.progressBar.setOrientation(SwingConstants.HORIZONTAL);
        this.progressBar.setMinimum(0);
        this.progressBar.setMaximum(99);
        this.progressBar.setValue(0);

        JPanel statusBar = new JPanel();
        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();
        statusBar.setLayout(layout);

        constraints.weightx = 1.0;
        constraints.fill = GridBagConstraints.HORIZONTAL;
        statusBar.add(this.sysMessage, constraints);

        constraints.weightx = 0.0;
        constraints.fill = GridBagConstraints.NONE;
        constraints.insets = new Insets(2, 2, 2, 2);
        statusBar.add(this.progressBar, constraints);

        return statusBar;
    }

    /**
     * 国村選択ツリービューを返す。
     * @return 国村選択ツリービュー
     */
    public JTree getTreeView(){
        return this.landsTreeView.getTreeView();
    }

    /**
     * タブビューを返す。
     * @return タブビュー
     */
    public TabBrowser getTabBrowser(){
        return this.tabBrowser;
    }

    /**
     * 村一覧ビューを返す。
     * @return 村一番ビュー
     */
    public LandsTree getLandsTree(){
        return this.landsTreeView;
    }
    
    /**
     * フレームタイトルを設定する。
     * タイトルは指定された村名 + " - Jindolf"
     * @param villageName 村の名前
     */
    public void setFrameTitle(CharSequence villageName){
        String title = Jindolf.TITLE;

        if(villageName != null && villageName.length() > 0){
            title = villageName + " - " + title;
        }

        setTitle(title);

        return;
    }

    /**
     * プログレスバーとカーソルの設定を行う。
     * @param busy trueならプログレスバーのアニメ開始&WAITカーソル。
     *              falseなら停止&通常カーソル。
     */
    public void setBusy(final boolean busy){
        Runnable microJob = new Runnable(){
            public void run(){
                Cursor cursor;
                if(busy){
                    cursor = Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR);
                }else{
                    cursor = Cursor.getDefaultCursor();
                }
                
                Component glass = getGlassPane();
                glass.setCursor(cursor);
                glass.setVisible(busy == true);
                progressBar.setIndeterminate(busy == true);
                
                return;
            }
        };

        if(SwingUtilities.isEventDispatchThread()){
            microJob.run();
        }else{
            try{
                SwingUtilities.invokeAndWait(microJob);
            }catch(Exception e){
                Jindolf.logger.log(Level.SEVERE, "ビジー処理で失敗", e);
            }
        }
        
        return;
    }

    /**
     * ステータスバーの更新
     * @param message 更新文字列
     */
    public void updateSysMessage(String message){
        if(message == null) return;
        if(message.length() <= 0) message = " ";
        this.sysMessage.setText(message);   // Thread safe
        GUIUtils.dispatchEmptyAWTEvent();
        return;
    }

    /**
     * 初期パネルを表示する。
     */
    public void showInitPanel(){
        cardLayout.show(cards, INITCARD);
        return;
    }
    
    /**
     * 村情報を表示する。
     * @param village 村
     */
    public void showVillageInfo(Village village){
        setFrameTitle(village.getVillageFullName());

        this.tabBrowser.setVillage(village);
        cardLayout.show(cards, BROWSECARD);
        this.tabBrowser.repaint();
        this.tabBrowser.revalidate();

        return;
    }

    /**
     * 国情報を表示する。
     * @param land 国
     */
    public void showLandInfo(Land land){
        setFrameTitle(land.getLandName());

        this.landInfo.removeAll();

        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();
        this.landInfo.setLayout(layout);

        constraints.insets = new Insets(2, 2, 2, 2);

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("国名 : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        this.landInfo.add(new JLabel(land.getLandName()), constraints);

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("識別名 : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        this.landInfo.add(new JLabel(land.getLandIdentifier()), constraints);

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("ベースURL : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        JTextField baseURL =
                new JTextField(land
                               .getServerAccess()
                               .getBaseURL()
                               .toString() );
        baseURL.setEditable(false);
        baseURL.setBorder(new EmptyBorder(0, 0, 0, 0));
        this.landInfo.add(baseURL, constraints);

        Date start = land.getStartDate();
        Date end = land.getEndDate();
        String startStr;
        String endStr;
        synchronized(dform){
            if(start.equals(Land.DATE_UNKNOWN)){
                startStr = "(不明)";
            }else{
                startStr = dform.format(start);
            }
            if(end.equals(Land.DATE_UNKNOWN)){
                endStr = "まだまだ";
            }else{
                endStr = dform.format(end);
            }
        }

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("建国 : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        this.landInfo.add(new JLabel(startStr), constraints);

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("亡国 : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        this.landInfo.add(new JLabel(endStr), constraints);

        String status;
        switch(land.getType()){
        case CLOSED:
            status = "サービス終了";
            break;
        case HISTORICAL:
            status = "過去ログ提供のみ";
            break;
        case ACTIVE:
            status = "稼動中";
            break;
        default:
            assert false;
            status = "";
            break;
        }

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("状態 : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        this.landInfo.add(new JLabel(status), constraints);

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("ロケール : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        this.landInfo.add(new JLabel(
                land
                .getLocale()
                .toString()
                ), constraints);

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("時間帯 : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        this.landInfo.add(new JLabel(
                land
                .getTimeZone()
                .getDisplayName()
                ), constraints);

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("連絡先 : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        JTextField contact = new JTextField(land.getContactInfo());
        contact.setEditable(false);
        contact.setBorder(new EmptyBorder(0, 0, 0, 0));
        this.landInfo.add(contact, constraints);

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        this.landInfo.add(new JLabel("説明 : "), constraints);
        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        this.landInfo.add(new JLabel(land.getDescription()), constraints);

        constraints.fill = GridBagConstraints.BOTH;
        constraints.weightx = 1.0;
        constraints.weighty = 1.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.gridheight = GridBagConstraints.REMAINDER;
        this.landInfo.add(new JPanel(), constraints);  // ダミー詰め物

        cardLayout.show(cards, LANDCARD);
        this.landInfo.repaint();
        this.landInfo.revalidate();

        return;
    }
    
    // TODO setEnabled()を全子フレームにも波及させるべきか
}
