/*
 * 会話部描画
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: TalkDraw.java 211 2008-09-29 10:42:27Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.font.FontRenderContext;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.text.DateFormat;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * 会話部の描画。
 * 会話部描画領域は、キャプション部と発言部から構成される。
 */
public class TalkDraw implements RowsLtoR{

    public static final Color COLOR_PUBLIC   = new Color(0xffffff);
    public static final Color COLOR_WOLFONLY = new Color(0xff7777);
    public static final Color COLOR_PRIVATE  = new Color(0x939393);
    public static final Color COLOR_GRAVE    = new Color(0x9fb7cf);
                        
    private static final int BALOONTIP_WIDTH = 16;
    private static final int BALOONTIP_HEIGHT = 8;
    private static final int UNDER_MARGIN = 15;
    private static final int OFFSET_ANCHOR = 36;

    private static Color COLOR_TRANS = new Color(0, 0, 0, 0);
    private static int BALOON_R = 10;
    private static BufferedImage BALOON_PUBLIC;
    private static BufferedImage BALOON_WOLFONLY;
    private static BufferedImage BALOON_GRAVE;
    private static BufferedImage BALOON_PRIVATE;
   
    private static float ANCHOR_FONT_RATIO = 0.9f;
    
    static{
        BALOON_PUBLIC   = createWedgeImage(COLOR_PUBLIC);
        BALOON_WOLFONLY = createBubbleImage(COLOR_WOLFONLY);
        BALOON_PRIVATE  = createBubbleImage(COLOR_PRIVATE);
        BALOON_GRAVE    = createBubbleImage(COLOR_GRAVE);
    }
    
    /**
     * 指定した色で描画したクサビイメージを取得する。
     * @param color 色
     * @return クサビイメージ
     */
    private static BufferedImage createWedgeImage(Color color){
        BufferedImage image;
        image = new BufferedImage(BALOONTIP_WIDTH,
                                  BALOONTIP_HEIGHT,
                                  BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2 = image.createGraphics();
        RenderingHints renderHints = GUIUtils.getQualityHints();
        g2.addRenderingHints(renderHints);
        g2.setColor(COLOR_TRANS);
        g2.fillRect(0, 0, BALOONTIP_WIDTH, BALOONTIP_HEIGHT);
        g2.setColor(color);
        Polygon poly = new Polygon();
        poly.addPoint(8, 8);
        poly.addPoint(16, 8);
        poly.addPoint(16, 0);
        g2.fillPolygon(poly);
        return image;
    }
    
    /**
     * 指定した色で描画した泡イメージを取得する。
     * @param color 色
     * @return 泡イメージ
     */
    private static BufferedImage createBubbleImage(Color color){
        BufferedImage image;
        image = new BufferedImage(BALOONTIP_WIDTH,
                                  BALOONTIP_HEIGHT,
                                  BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2 = image.createGraphics();
        RenderingHints renderHints = GUIUtils.getQualityHints();
        g2.addRenderingHints(renderHints);
        g2.setColor(COLOR_TRANS);
        g2.fillRect(0, 0, BALOONTIP_WIDTH, BALOONTIP_HEIGHT);
        g2.setColor(color);
        g2.fillOval(2, 4, 4, 4);
        g2.fillOval(8, 2, 6, 6);
        return image;
    }
    
    /**
     * 会話表示用フォントからアンカー表示用フォントを派生させる。
     * @param font 派生元フォント
     * @return 派生先フォント
     */
    private static Font getAnchorFont(Font font){
        float fontSize = font.getSize2D();
        float newSize = fontSize * ANCHOR_FONT_RATIO;
        return font.deriveFont(newSize);
    }
    
    private final Talk talk;
    private Anchor showingAnchor;
    private GlyphDraw caption;
    private GlyphDraw dialog;
    private List<AnchorDraw> anchorTalks = new LinkedList<AnchorDraw>();
    private Image faceImage;
    private Point imageOrigin;
    private Point dialogOrigin;
    private Point tipOrigin;
    private final Rectangle bounds = new Rectangle();
    private Font font;
    private Font anchorFont;
    private FontRenderContext frc;

    /**
     * コンストラクタ
     * @param talk 一発言
     * @param font フォント
     * @param frc フォント描画設定
     */
    public TalkDraw(Talk talk, Font font, FontRenderContext frc){
        super();

        this.talk = talk;
        this.font = font;
        this.frc = frc;
        this.anchorFont = getAnchorFont(this.font);
        
        Village village = this.talk.getPeriod().getVillage();
        Avatar avatar = this.talk.getAvatar();
        
        Image image;
        if(talk.getTalkType() == Talk.Type.GRAVE){
            image = village.getGraveImage();
        }else{
            image = village.getAvatarFaceImage(avatar);
        }
        this.faceImage = image;
        
        this.caption = new GlyphDraw(this.font, this.frc, getCaptionString());
        this.dialog = new GlyphDraw(this.font, this.frc, this.talk.getDialog());
        this.caption.setColor(Color.WHITE);
        this.dialog.setColor(Color.BLACK);
        
        Period period = this.talk.getPeriod();
        List<Anchor> anchorList = Anchor.getAnchorList(this.talk.getDialog(),
                                                       period.getDay() );
        this.dialog.setAnchorSet(anchorList);
        
        return;
    }

    /**
     * Talk取得
     * @return Talkインスタンス
     */
    public Talk getTalk(){
        return this.talk;
    }
    
    /**
     * キャプション文字列を取得する。
     * @return キャプション文字列
     */
    private CharSequence getCaptionString(){
        StringBuilder result = new StringBuilder();
        
        Avatar avatar = this.talk.getAvatar();
        
        result.append(avatar.getFullName()).append(' ');
        int hour = this.talk.getHour();
        if(hour <= 11) result.append("午前 ").append(hour);
        else           result.append("午後 ").append(hour-12);
        result.append("時").append(' ');
        int minute = this.talk.getMinute();
        result.append(minute).append("分").append(' ');
        result.append(this.talk.getAnchorNotation()).append('\n');

        DateFormat dform =
            DateFormat.getDateTimeInstance(DateFormat.MEDIUM,
                                           DateFormat.MEDIUM);
        long epoch = this.talk.getTimeFromID();
        String decoded = dform.format(epoch);
        result.append(decoded);
        
        int count = this.talk.getTalkCount();
        if(count > 0){
            Talk.Type type = this.talk.getTalkType();
            result.append(" (").append(Talk.encodeColorName(type));
            result.append('#').append(count).append(')');
        }
        
        int charNum = this.talk.getTotalChars();
        if(charNum > 0){
            result.append(' ').append(charNum).append('字');
        }
        
        return result;
    }
    
    /**
     * 会話部背景色を返す。
     * @return 会話部背景色
     */
    protected Color getTalkColor(){
        Color result;
        switch(this.talk.getTalkType()){
        case PUBLIC:
            result = COLOR_PUBLIC;
            break;
        case WOLFONLY:
            result = COLOR_WOLFONLY;
            break;
        case PRIVATE:
            result = COLOR_PRIVATE;
            break;
        case GRAVE:
            result = COLOR_GRAVE;
            break;
        default:
            assert false;
            result = null;
        }
        return result;
    }
    
    /**
     * 新しい幅を指定し、寸法の再計算、内部の再レイアウトを促す。
     * @param newWidth 新しいピクセル幅
     * @return 新しい寸法
     */
    public Rectangle setWidth(int newWidth){
        int imageWidth = this.faceImage.getWidth(null);
        int imageHeight = this.faceImage.getHeight(null);
        int tipWidth = BALOON_WOLFONLY.getWidth();

        int minWidth = imageWidth + tipWidth + BALOON_R * 2;
        if(newWidth < minWidth) newWidth = minWidth;
        
        this.caption.setWidth(newWidth);
        int captionWidth  = this.caption.getWidth();
        int captionHeight = this.caption.getHeight();

        this.dialog.setWidth(newWidth - minWidth);
        int dialogWidth  = this.dialog.getWidth();
        int dialogHeight = this.dialog.getHeight();
        
        int baloonWidth  = dialogWidth + BALOON_R * 2;
        int baloonHeight = dialogHeight + BALOON_R * 2;
        
        int imageAndDialogWidth = imageWidth + tipWidth + baloonWidth;
        
        int totalWidth = Math.max(captionWidth, imageAndDialogWidth);

        int totalHeight = captionHeight;
        totalHeight += Math.max(imageHeight, baloonHeight);
        
        int imageYpos = captionHeight;
        int dialogYpos = captionHeight;
        int tipYpos = captionHeight;
        if(imageHeight < baloonHeight){
            imageYpos += (baloonHeight - imageHeight) / 2;
            tipYpos += (baloonHeight - BALOON_WOLFONLY.getHeight()) / 2;
            dialogYpos += BALOON_R;
        }else{
            dialogYpos += (imageHeight - baloonHeight) / 2 + BALOON_R;
            tipYpos += (imageHeight - BALOON_WOLFONLY.getHeight()) / 2;
        }
        
        this.imageOrigin = new Point(0, imageYpos);
        this.caption.setPos(this.bounds.x + 0, this.bounds.y + 0);
        this.dialogOrigin = new Point(imageWidth+tipWidth+BALOON_R, dialogYpos);
        this.dialog.setPos(this.bounds.x + imageWidth+tipWidth+BALOON_R,
                           this.bounds.y + dialogYpos);
        this.tipOrigin = new Point(imageWidth, tipYpos);

        for(AnchorDraw anchorDraw : this.anchorTalks){
            anchorDraw.setWidth(newWidth - OFFSET_ANCHOR);
            totalHeight += anchorDraw.getHeight();
        }
        
        this.bounds.width = totalWidth;
        this.bounds.height = totalHeight + UNDER_MARGIN;
        
        return this.bounds;
    }

    /**
     * 描画を行う。
     * @param g グラフィックコンテキスト
     */
    public void paint(Graphics2D g){
        final int xPos = this.bounds.x;
        final int yPos = this.bounds.y;
        
        this.caption.paint(g);
        
        g.drawImage(this.faceImage,
                    xPos + this.imageOrigin.x,
                    yPos + this.imageOrigin.y,
                    null );
        
        BufferedImage tip;
        switch(this.talk.getTalkType()){
        case WOLFONLY:
            tip = BALOON_WOLFONLY;
            break;
        case PUBLIC:
            tip = BALOON_PUBLIC;
            break;
        case GRAVE:
            tip = BALOON_GRAVE;
            break;
        case PRIVATE:
            tip = BALOON_PRIVATE;
            break;
        default:
            tip = null;
            assert false;
            break;
        }
        g.drawImage(tip,
                    xPos + this.tipOrigin.x,
                    yPos + this.tipOrigin.y,
                    null );
        
        int baloonWidth  = this.dialog.getWidth()  + (BALOON_R * 2);
        int baloonHeight = this.dialog.getHeight() + (BALOON_R * 2);
        g.setColor(getTalkColor());
        g.fillRoundRect(
                xPos + this.dialogOrigin.x - BALOON_R,
                yPos + this.dialogOrigin.y - BALOON_R,
                baloonWidth,
                baloonHeight,
                BALOON_R,
                BALOON_R );
        
        this.dialog.paint(g);
        
        int anchorX = xPos + OFFSET_ANCHOR;
        int anchorY = yPos + this.dialogOrigin.y + baloonHeight;
        
        for(AnchorDraw anchorDraw : this.anchorTalks){
            anchorDraw.setPos(anchorX, anchorY);
            anchorDraw.paint(g);
            anchorY += anchorDraw.getHeight();
        }
        
        return;
    }
    
    /**
     * 描画領域の寸法を返す。
     * @return 描画領域の寸法
     */
    public Rectangle getBounds(){
        return this.bounds;
    }

    /**
     * 描画開始位置の指定。
     * @param xPos 描画開始位置のx座標
     * @param yPos 描画開始位置のy座標
     */
    public void setPos(int xPos, int yPos){
        this.bounds.x = xPos;
        this.bounds.y = yPos;
        return;
    }

    /**
     * 描画領域の寸法幅を返す。
     * @return 描画領域の寸法幅
     */
    public int getWidth(){
        return this.bounds.width;
    }

    /**
     * 描画領域の寸法高を返す。
     * @return 描画領域の寸法高
     */
    public int getHeight(){
        return this.bounds.height;
    }
    
    /**
     * フォント描画設定を変更する。
     * @param font フォント
     * @param frc フォント描画設定
     */
    public void setFontInfo(Font font, FontRenderContext frc){
        this.font = font;
        this.frc = frc;
        this.anchorFont = getAnchorFont(this.font);

        this.caption.setFontInfo(this.font, this.frc);
        this.dialog.setFontInfo(this.font, this.frc);

        for(AnchorDraw anchorDraw : this.anchorTalks){
            anchorDraw.setFontInfo(this.anchorFont, this.frc);
        }
        
        int width = getWidth();
        setWidth(width);
        
        return;
    }
    
    /**
     * ドラッグ処理を行う。
     * @param from ドラッグ開始位置
     * @param to 現在のドラッグ位置
     */
    public void drag(Point from, Point to){
        this.caption.drag(from, to);
        this.dialog.drag(from, to);
        for(AnchorDraw anchorDraw : this.anchorTalks){
            anchorDraw.drag(from, to);
        }
        return;
    }

    /**
     * 受け取った文字列に選択文字列を追加する。
     * @param appendable 追加対象文字列
     * @return 引数と同じインスタンス
     * @throws java.io.IOException
     */
    public Appendable appendSelected(Appendable appendable)
            throws IOException{
        this.caption.appendSelected(appendable);
        this.dialog .appendSelected(appendable);

        for(AnchorDraw anchorDraw : this.anchorTalks){
            anchorDraw.appendSelected(appendable);
        }

        return appendable;
    }
    
    /**
     * 選択範囲の解除。
     */
    public void clearSelect(){
        this.caption.clearSelect();
        this.dialog.clearSelect();
        for(AnchorDraw anchorDraw : this.anchorTalks){
            anchorDraw.clearSelect();
        }
        return;
    }
    
    /**
     * 与えられた座標にアンカー文字列が存在すればAnchorを返す。
     * @param pt 座標
     * @return アンカー
     */
    public Anchor getAnchor(Point pt){
        Anchor result = this.dialog.getAnchor(pt);
        return result;
    }
    
    /**
     * アンカーを展開表示する。
     * アンカーにnullを指定すればアンカー表示は非表示となる。
     * @param anchor アンカー
     * @param talkList アンカーの示す一連のTalk
     */
    public void showAnchorTalks(Anchor anchor, List<Talk> talkList){
        if(anchor == null || this.showingAnchor == anchor){
            this.showingAnchor = null;
            this.anchorTalks.clear();
            int width = getWidth();
            setWidth(width);
            return;
        }
        
        this.showingAnchor = anchor;
        
        this.anchorTalks.clear();
        for(Talk anchorTalk : talkList){
            AnchorDraw anchorDraw = new AnchorDraw(anchorTalk,
                                                   this.anchorFont,
                                                   this.frc );
            anchorDraw.setFontInfo(this.anchorFont, this.frc);
            this.anchorTalks.add(anchorDraw);
        }
        
        int width = getWidth();
        setWidth(width);
        
        return;
    }
    
    /**
     * 与えられた座標に検索マッチ文字列があればそのインデックスを返す。
     * @param pt 座標
     * @return 検索マッチインデックス
     */
    public int getRegexMatchIndex(Point pt){
        int index = this.dialog.getRegexMatchIndex(pt);
        return index;
    }
    
    /**
     * 検索文字列パターンを設定する
     * @param searchRegex パターン
     * @return ヒット数
     */
    public int setRegex(Pattern searchRegex){
        int total = 0;
        
        total += this.dialog.setRegex(searchRegex);
/*
        for(AnchorDraw anchorDraw : this.anchorTalks){
            total += anchorDraw.setRegex(searchRegex);
        }
*/ // TODO よくわからんので保留        
        return total;
    }
    
    /**
     * 検索ハイライトインデックスを返す。
     * @return 検索ハイライトインデックス。見つからなければ-1。
     */
    public int getHotTargetIndex(){
        return this.dialog.getHotTargetIndex();
    }
    
    /**
     * 検索ハイライトを設定する。
     * @param index ハイライトインデックス。負ならハイライト全クリア。 
     */
    public void setHotTargetIndex(int index){
        this.dialog.setHotTargetIndex(index);
        return;
    }
    
    /**
     * 検索一致件数を返す。
     * @return 検索一致件数
     */
    public int getRegexMatches(){
        return this.dialog.getRegexMatches();
    }
    
    /**
     * 特別な検索ハイライト描画をクリアする。
     */
    public void clearHotTarget(){
        this.dialog.clearHotTarget();
        return;
    }
    
    /**
     * 特別な検索ハイライト領域の寸法を返す。
     * @return ハイライト領域寸法
     */
    public Rectangle getHotTargetRectangle(){
        return this.dialog.getHotTargetRectangle();
    }
}
