/*
 * システムイベントメッセージの描画
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: SysEventDraw.java 154 2008-09-04 16:15:41Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.font.FontRenderContext;
import java.io.IOException;

/**
 * システムイベントメッセージの描画。
 */
public class SysEventDraw implements RowsLtoR{

    public static final Color COLOR_ANNOUNCE = new Color(0xffffff);
    public static final Color COLOR_ORDER    = new Color(0xf04040);
    public static final Color COLOR_EXTRA    = new Color(0x808080);
    
    private static final int INSET = 10;
    private static final int UNDER_MARGIN = 15;
    
    private final Rectangle bounds = new Rectangle();
    private final SysEvent sysEvent;
    private final GlyphDraw sysMessage;

    /**
     * コンストラクタ
     * @param sysEvent システムイベント
     * @param font フォント
     * @param frc フォント描画設定
     */
    public SysEventDraw(SysEvent sysEvent, Font font, FontRenderContext frc){
        super();
        this.sysEvent = sysEvent;
        
        this.sysMessage = new GlyphDraw(font, frc, this.sysEvent.getMessage());
        this.sysMessage.setColor(getEventColor());
        
        return;
    }
    
    /**
     * システムイベントの取得
     * @return システムイベント
     */
    public SysEvent getSysEvent(){
        return this.sysEvent;
    }
    
    /**
     * イベント種別に応じた前景色を返す。
     * @return イベント種別前景色
     */
    protected Color getEventColor(){
        Color result;
        switch(this.sysEvent.getEventClass()){
        case ANNOUNCE:
            result = COLOR_ANNOUNCE;
            break;
        case ORDER:
            result = COLOR_ORDER;
            break;
        case EXTRA:
            result = COLOR_EXTRA;
            break;
        default:
            assert false;
            result = null;
        }
        return result;
    }
    
    /**
     * 描画領域の寸法を返す。
     * @return 描画領域の寸法
     */
    public Rectangle getBounds(){
        return this.bounds;
    }

    /**
     * 描画領域の寸法幅を返す。
     * @return 描画領域の寸法幅
     */
    public int getWidth(){
        return this.bounds.width;
    }

    /**
     * 描画領域の寸法高を返す。
     * @return 描画領域の寸法高
     */
    public int getHeight(){
        return this.bounds.height;
    }

    /**
     * 描画開始位置の指定。
     * @param xPos 描画開始位置のx座標
     * @param yPos 描画開始位置のy座標
     */
    public void setPos(int xPos, int yPos){
        this.bounds.x = xPos;
        this.bounds.y = yPos;
    }

    /**
     * 新しい幅を指定し、寸法の再計算、内部の再レイアウトを促す。
     * @param newWidth 新しいピクセル幅
     * @return 新しい寸法
     */
    public Rectangle setWidth(int newWidth){
        Rectangle child;
        child = this.sysMessage.setWidth(newWidth-INSET-INSET);
        this.bounds.width = newWidth;
        this.bounds.height = child.height + INSET + INSET + UNDER_MARGIN;
        return this.bounds;
    }

    /**
     * フォント描画設定を変更する。
     * @param font フォント
     * @param frc フォント描画設定
     */
    public void setFontInfo(Font font, FontRenderContext frc){
        this.sysMessage.setFontInfo(font, frc);
        int width = getWidth();
        setWidth(width);
        return;
    }
    
    /**
     * ドラッグ処理を行う。
     * @param from ドラッグ開始位置
     * @param to 現在のドラッグ位置
     */
    public void drag(Point from, Point to){
        this.sysMessage.drag(from, to);
        return;
    }

    /**
     * 受け取った文字列に選択文字列を追加する。
     * @param appendable 追加対象文字列
     * @return 引数と同じインスタンス
     * @throws java.io.IOException
     */
    public Appendable appendSelected(Appendable appendable)
            throws IOException{
        this.sysMessage.appendSelected(appendable);
        return appendable;
    }
    
    /**
     * 選択範囲の解除。
     */
    public void clearSelect(){
        this.sysMessage.clearSelect();
        return;
    }

    /**
     * 描画を行う。
     * @param g グラフィックコンテキスト
     */
    public void paint(Graphics2D g){
        g.setColor(getEventColor());
        
        g.drawRect(this.bounds.x,
                   this.bounds.y,
                   this.bounds.width,
                   this.bounds.height - UNDER_MARGIN);

        this.sysMessage.setPos(this.bounds.x + INSET, this.bounds.y + INSET);
        this.sysMessage.paint(g);
        return;
    }
}
