/*
 * discussion browser
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Discussion.java 238 2008-10-09 16:31:29Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.font.FontRenderContext;
import java.io.IOException;
import java.util.EventListener;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.regex.Pattern;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.Scrollable;
import javax.swing.SwingConstants;
import javax.swing.event.EventListenerList;
import javax.swing.event.MouseInputListener;
import javax.swing.text.DefaultEditorKit;

/**
 * 発言表示画面
 */
@SuppressWarnings("serial")
public class Discussion extends JComponent
        implements Scrollable, MouseInputListener{

    private static final Insets insets = new Insets(15, 15, 15, 15);
    private static final int horizontalGap = insets.left + insets.right;
    private static final int verticalGap = insets.top + insets.bottom;
    
    private Period period;
    private TopicFilter topicFilter = null;
    private TopicFilter.FilterContext filterContext;
    private RegexPattern regexPattern = null;
    private final List<RowsLtoR> rowList = new LinkedList<RowsLtoR>();
    private Dimension idealSize;
    private int lastWidth = -1;
    private Point lastPressedPoint;
    private Point dragFrom;
    private DiscussionPopup popup = new DiscussionPopup();
    private Font font;
    private FontRenderContext renderContext;
    private final RenderingHints hints = new RenderingHints(null);
    
    private final EventListenerList thisListenerList = new EventListenerList();

    private Action copySelectedAction =
            new ProxyAction(ActionManager.COMMAND_COPY);
    
    /**
     * 初期Periodを指定してメインブラウザを作成する。
     * @param period 初期Period
     */
    public Discussion(Period period,
                       Font font,
                       FontRenderContext renderContext){
        super();

        this.font          = font;
        this.renderContext = renderContext;

        this.hints.put(RenderingHints.KEY_ANTIALIASING,
                       RenderingHints.VALUE_ANTIALIAS_ON);
        updateRenderingHints();
        
        setPeriod(period);

        addMouseListener(this);
        addMouseMotionListener(this);

        setComponentPopupMenu(this.popup);

        updateInputMap();
        ActionMap actionMap = getActionMap();
        actionMap.put(DefaultEditorKit.copyAction, this.copySelectedAction);

        return;
    }

    /**
     * 描画ヒントの更新。
     */
    private void updateRenderingHints(){
        boolean isAntiAliased = this.renderContext.isAntiAliased();
        boolean usesFractionalMetrics =
                this.renderContext.usesFractionalMetrics();
        
        Object textAliaseValue = RenderingHints.VALUE_TEXT_ANTIALIAS_OFF;
        Object textFractionalValue =
                RenderingHints.VALUE_FRACTIONALMETRICS_OFF;
        
        if(isAntiAliased){
            textAliaseValue = RenderingHints.VALUE_TEXT_ANTIALIAS_ON;
        }
        if(usesFractionalMetrics){
            textFractionalValue = RenderingHints.VALUE_FRACTIONALMETRICS_ON;
        }

        this.hints.put(RenderingHints.KEY_TEXT_ANTIALIASING,
                       textAliaseValue);
        this.hints.put(RenderingHints.KEY_FRACTIONALMETRICS,
                       textFractionalValue);
        
        return;
    }
    
    /**
     * フォント描画設定を変更する。
     * @param font フォント
     * @param renderContext フォント描画設定
     */
    public void setFontInfo(Font font, FontRenderContext renderContext){
        this.font          = font;
        this.renderContext = renderContext;
        
        updateRenderingHints();
        
        for(RowsLtoR row : this.rowList){
            row.setFontInfo(this.font, this.renderContext);
        }

        layoutRows();
        
        return;
    }
    
    /**
     * 現在のPeriodを返す。
     * @return 現在のPeriod
     */
    public Period getPeriod(){
        return this.period;
    }

    /**
     * Periodを更新する。
     * 古いPeriodの表示内容は消える。
     * 新しいPeriodの表示内容はまだ反映されない。
     * @param period 新しいPeriod
     */
    public void setPeriod(Period period){
        if(period == null){
            this.period = null;
            this.rowList.clear();
            return;
        }
        
        if(   this.period == period
           && period.getTopics() == this.rowList.size() ){
            return;
        }

        this.period = period;

        this.filterContext = null;

        this.rowList.clear();
        for(Topic topic : this.period.getTopicList()){
            RowsLtoR row;
            if(topic instanceof Talk){
                Talk talk = (Talk) topic;
                row = new TalkDraw(talk, this.font, this.renderContext);
            }else if(topic instanceof SysEvent){
                SysEvent sysEvent = (SysEvent) topic;
                row = new SysEventDraw(sysEvent, this.font, this.renderContext);
            }else{
                assert false;
                continue;
            }
            this.rowList.add(row);
        }

        clearSizeCache();
        int width = getWidth();
        calcIdealSize(width);
        
        repaint();
        revalidate();
        
        return;
    }

    /**
     * インセットを返す。
     * @return インセット
     */
    @Override
    public Insets getInsets(){
        return insets;
    }
    
    /**
     * フィルタを適用してPeriodの内容を出力する。
     */
    public void showTopics(){
        setPeriod(getPeriod()); // TODO Periodからのイベントを受け取るべし？
        return;
    }

    /**
     * 発言フィルタを設定する。
     * @param filter
     */
    public void setTopicFilter(TopicFilter filter){
        this.topicFilter = filter;
        filtering();
        return;
    }

    /**
     * 発言フィルタを適用する。
     */
    public void filtering(){
        if(   this.topicFilter != null
           && this.topicFilter.isSame(this.filterContext)){
            return;
        }

        if(this.topicFilter != null){
            this.filterContext = this.topicFilter.getFilterContext();
        }else{
            this.filterContext = null;
        }

        layoutRows();

        clearSelect();
        
        return;
    }

    /**
     * 検索パターンを取得する。
     * @return 検索パターン
     */
    public RegexPattern getRegexPattern(){
        return this.regexPattern;
    }
    
    /**
     * 検索パターンを設定する。
     * @param regexPattern 検索パターン
     */
    public void setRegexPattern(RegexPattern regexPattern){
        this.regexPattern = regexPattern;
        return;
    }

    /**
     * 与えられた正規表現にマッチする文字列をハイライト描画する。
     * @return ヒット件数
     */
    public int highlightRegex(){
        int total = 0;

        clearHotTarget();
        
        Pattern pattern = null;
        if(this.regexPattern != null){
            pattern = this.regexPattern.getPattern();
        }
        
        for(RowsLtoR row : this.rowList){
            if( ! (row instanceof TalkDraw) ) continue;
            TalkDraw talkDraw = (TalkDraw) row;
            total += talkDraw.setRegex(pattern);
        }
        
        repaint();
        
        return total;
    }
    
    /**
     * 検索結果の次候補をハイライト表示する。
     */
    public void nextHotTarget(){
        TalkDraw oldTalk = null;
        int oldIndex = -1;
        TalkDraw newTalk = null;
        int newIndex = -1;
        TalkDraw firstTalk = null;
        
        boolean findOld = true;
        for(RowsLtoR row : this.rowList){
            if( ! (row instanceof TalkDraw) ) continue;
            TalkDraw talkDraw = (TalkDraw) row;
            int matches = talkDraw.getRegexMatches();
            if(firstTalk == null && matches > 0){
                firstTalk = talkDraw;
            }
            if(findOld){
                int index = talkDraw.getHotTargetIndex();
                if(index < 0) continue;
                oldTalk = talkDraw;
                oldIndex = index;
                scrollRectWithMargin(talkDraw.getHotTargetRectangle());
                if(oldIndex < matches-1 && ! isFiltered(row) ){
                    newTalk = talkDraw;
                    newIndex = oldIndex + 1;
                    break;
                }
                findOld = false;
            }else{
                if(isFiltered(row)) continue;
                if(matches <= 0) continue;
                newTalk = talkDraw;
                newIndex = 0;
                break;
            }
        }
        
        Rectangle showRect = null;
        if(oldTalk == null && firstTalk != null){
            firstTalk.setHotTargetIndex(0);
            showRect = firstTalk.getHotTargetRectangle();
        }else if(   oldTalk != null
                 && newTalk != null){
            oldTalk.clearHotTarget();
            newTalk.setHotTargetIndex(newIndex);
            showRect = newTalk.getHotTargetRectangle();
        }
        
        if(showRect != null){
            scrollRectWithMargin(showRect);
        }
        
        repaint();
        
        return;
    }
    
    /**
     * 検索結果の前候補をハイライト表示する。
     */
    public void prevHotTarget(){
        TalkDraw oldTalk = null;
        int oldIndex = -1;
        TalkDraw newTalk = null;
        int newIndex = -1;
        TalkDraw firstTalk = null;
        
        boolean findOld = true;
        int size = this.rowList.size();
        ListIterator<RowsLtoR> iterator = this.rowList.listIterator(size);
        while(iterator.hasPrevious()){
            RowsLtoR row = iterator.previous();
            if( ! (row instanceof TalkDraw) ) continue;
            TalkDraw talkDraw = (TalkDraw) row;
            int matches = talkDraw.getRegexMatches();
            if(firstTalk == null && matches > 0){
                firstTalk = talkDraw;
            }
            if(findOld){
                int index = talkDraw.getHotTargetIndex();
                if(index < 0) continue;
                oldTalk = talkDraw;
                oldIndex = index;
                scrollRectWithMargin(talkDraw.getHotTargetRectangle());
                if(oldIndex > 0 && ! isFiltered(row) ){
                    newTalk = talkDraw;
                    newIndex = oldIndex - 1;
                    break;
                }
                findOld = false;
            }else{
                if(isFiltered(row)) continue;
                if(matches <= 0) continue;
                newTalk = talkDraw;
                newIndex = matches-1;
                break;
            }
        }
        
        Rectangle showRect = null;
        if(oldTalk == null && firstTalk != null){
            int matches = firstTalk.getRegexMatches();
            firstTalk.setHotTargetIndex(matches-1);
            showRect = firstTalk.getHotTargetRectangle();
        }else if(   oldTalk != null
                 && newTalk != null){
            oldTalk.clearHotTarget();
            newTalk.setHotTargetIndex(newIndex);
            showRect = newTalk.getHotTargetRectangle();
        }
        
        if(showRect != null){
            scrollRectWithMargin(showRect);
        }
        
        repaint();
        
        return;
    }
    
    /**
     * 検索結果の特殊ハイライト表示を解除
     */
    public void clearHotTarget(){
        for(RowsLtoR row : this.rowList){
            if( ! (row instanceof TalkDraw) ) continue;
            TalkDraw talkDraw = (TalkDraw) row;
            talkDraw.clearHotTarget();
        }
        repaint();
        return;
    }
    
    /**
     * 指定した領域に若干の上下マージンを付けてスクロールウィンドウに表示する。
     * @param rectangle 指定領域
     */
    private void scrollRectWithMargin(Rectangle rectangle){
        final int MARGINTOP    =  50;
        final int MARGINBOTTOM = 100;
        
        Rectangle show = new Rectangle(rectangle);
        show.y      -= MARGINTOP;
        show.height += MARGINTOP + MARGINBOTTOM;
        
        scrollRectToVisible(show);
        
        return;
    }
    
    /**
     * 過去に計算した寸法を破棄する。
     */
    private void clearSizeCache(){
        this.idealSize = null;
        this.lastWidth = -1;
        revalidate();
    }
    
    /**
     * 指定した矩形がフィルタリング対象か判定する。
     * @param row 矩形
     * @return フィルタリング対象ならtrue
     */
    private boolean isFiltered(RowsLtoR row){
        if(this.topicFilter == null) return false;
        
        Topic topic;
        if(row instanceof TalkDraw){
            topic = ((TalkDraw)row).getTalk();
        }else if(row instanceof SysEventDraw){
            topic = ((SysEventDraw)row).getSysEvent();
        }else{
            return false;
        }
        
        return this.topicFilter.isFiltered(topic);
    }
    
    /**
     * 指定位置から指定幅で描画した場合の寸法を返す。
     * @param xPos 指定位置X座標
     * @param yPos 指定位置Y座標
     * @param widthLimit 指定幅ピクセル数
     * @return 寸法
     */
    private Rectangle calcRect(int xPos, int yPos, int widthLimit){
        final int innerLimit = widthLimit - horizontalGap;
        Rectangle unionRect = null;
        for(RowsLtoR row : this.rowList){

            if(isFiltered(row)) continue;
            
            row.setPos(xPos, yPos);
            Rectangle rowBound = row.setWidth(innerLimit);
            yPos += rowBound.height;
            
            if(unionRect == null){
                unionRect = new Rectangle(rowBound);
            }else{
                unionRect.add(rowBound);
            }
        }
        return unionRect;
    }
    
    /**
     * Rowsの縦位置を再レイアウトする。
     */
    public void layoutRows(){
        int width = getWidth();
        calcIdealSizeIntl(width);
        repaint();
        revalidate();
        return;
    }
    
    /**
     * 指定された幅を満たす寸法でPreferredSizeを更新する。
     * @param newWidth 指定幅ピクセル数
     */
    private void calcIdealSize(final int newWidth){
        if(newWidth == this.lastWidth) return;
        calcIdealSizeIntl(newWidth);
    }
    
    /**
     * 指定された幅を満たす寸法でPreferredSizeを更新する。
     * @param newWidth 指定幅ピクセル数
     */
    private void calcIdealSizeIntl(final int newWidth){
        this.lastWidth = newWidth;

        Rectangle unionRect = calcRect(insets.left,
                                       insets.top,
                                       newWidth );
        
        if(this.idealSize == null){
            this.idealSize = new Dimension();
        }

        int newHeight = verticalGap;
        if(unionRect != null){
            newHeight += unionRect.height;
        }

        this.idealSize.width  = newWidth;
        this.idealSize.height = newHeight;
        
        setPreferredSize(this.idealSize);
        
        return;
    }
    
    /**
     * コンポーネント内を描画。
     * @param g グラフィックコンテキスト
     */
    @Override
    public void paintComponent(Graphics g){
        Graphics2D g2 = (Graphics2D) g;
        g2.setRenderingHints(this.hints);
        
        g2.setColor(Color.BLACK);
        Rectangle clipRect = g.getClipBounds();
        g2.fillRect(clipRect.x, clipRect.y, clipRect.width, clipRect.height);

        for(RowsLtoR row : this.rowList){
            Rectangle rowRect = row.getBounds();
            if( ! rowRect.intersects(clipRect) ) continue;
            
            if(isFiltered(row)) continue;
            
            row.paint(g2);
        }
        
        return;
    }

    /**
     * コンポーネントのサイズを設定する。
     * @param size
     */
    @Override
    public void setSize(Dimension size){
        setBounds(getX(), getY(), size.width, size.height);
        return;
    }
    
    /**
     * コンポーネントのサイズを設定する。
     * @param width
     * @param height
     */
    @Override
    public void setSize(int width, int height){
        setBounds(getX(), getY(), width, height);
        return;
    }
    
    /**
     * コンポーネントのサイズを設定する。
     * @param size
     * @deprecated
     */
    @Override
    @Deprecated
    public void resize(Dimension size){
        setBounds(getX(), getY(), size.width, size.height);
        return;
    }
    
    /**
     * コンポーネントのサイズを設定する。
     * @param width
     * @param height
     * @deprecated
     */
    @Override
    @Deprecated
    public void resize(int width, int height){
        setBounds(getX(), getY(), width, height);
        return;
    }

    /**
     * コンポーネントのサイズと位置を設定する。
     * @param rect
     */
    @Override
    public void setBounds(Rectangle rect){
        setBounds(rect.x, rect.y, rect.width, rect.height);
    }
    
    /**
     * コンポーネントのサイズと位置を設定する。
     * @param x
     * @param y
     * @param width
     * @param height
     */
    @Override
    public void setBounds(int x, int y, int width, int height){
        int oldWidth = getWidth();
        super.setBounds(x, y, width, height);
        if(oldWidth == width) return;

        calcIdealSize(width);
        if(   this.idealSize.width != width
           || this.idealSize.height != height ){
            revalidate();
        }
        
        return;
    }
    
    /**
     * ビューポートの推奨サイズを返す。
     * @return 推奨サイズ
     */
    public Dimension getPreferredScrollableViewportSize(){
        return getPreferredSize();
    }

    /**
     * ※ Scrollable参照
     * @return ビューポートの幅に合わせるならtrue
     */
    public boolean getScrollableTracksViewportWidth(){
        return true;
    }

    /**
     * ※ Scrollable参照
     * @return ビューポートの高さに合わせるならtrue
     */
    public boolean getScrollableTracksViewportHeight(){
        return false;
    }

    /**
     * ブロック単位のスクロール量を返す。
     * @param visibleRect ビューポート内可視領域
     * @param orientation スクロール方向
     * @param direction スクロール向き
     * @return スクロール量
     */
    public int getScrollableBlockIncrement(Rectangle visibleRect,
                                               int orientation,
                                               int direction        ){
        if(orientation == SwingConstants.VERTICAL){
            return visibleRect.height;
        }
        return 30; // TODO フォント高 × 1.5 ぐらい？
    }

    /**
     * 行単位のスクロール量を返す。
     * @param visibleRect ビューポート内可視領域
     * @param orientation スクロール方向
     * @param direction スクロール向き
     * @return スクロール量
     */
    public int getScrollableUnitIncrement(Rectangle visibleRect,
                                              int orientation,
                                              int direction      ){
        return 30;
    }
    
    /**
     * ドラッグ処理を行う。
     * @param from ドラッグ開始位置
     * @param to 現在のドラッグ位置
     */
    private void drag(Point from, Point to){
        Rectangle dragRegion = new Rectangle();
        dragRegion.setFrameFromDiagonal(from, to);
        
        for(RowsLtoR row : this.rowList){
            if(isFiltered(row)) continue;
            if( ! row.getBounds().intersects(dragRegion) ) continue;
            row.drag(from, to);
        }
        repaint();
        return;
    }
    
    /**
     * 選択範囲の解除。
     */
    private void clearSelect(){
        for(RowsLtoR row : this.rowList){
            row.clearSelect();
        }
        repaint();
        return;
    }
    
    /**
     * 与えられた点座標を包含する矩形を返す。
     * @param pt 点座標（JComponent基準）
     * @return 点座標を含む矩形。含む矩形がなければnullを返す。
     */
    public RowsLtoR hittedRow(Point pt){
        for(RowsLtoR row : this.rowList){
            if(isFiltered(row)) continue;
            Rectangle bounds = row.getBounds();
            if(bounds.contains(pt)) return row;
        }
        return null;
    }
    
    /**
     * アンカークリック動作の処理。
     * @param pt クリックポイント
     */
    private void hitAnchor(Point pt){
        if(this.period == null) return;

        RowsLtoR row = hittedRow(pt);
        if(row == null) return;
        if( ! (row instanceof TalkDraw) ) return;
        
        TalkDraw chat = (TalkDraw) row;
        Anchor anchor = chat.getAnchor(pt);
        if(anchor == null) return;

        for(AnchorHitListener listener : getAnchorHitListeners()){
            AnchorHitEvent event = new AnchorHitEvent(this, chat, anchor, pt);
            listener.anchorHitted(event);
        }
        
        return;
    }
    
    /**
     * 検索マッチ文字列クリック動作の処理。
     * @param pt クリックポイント
     */
    private void hitRegex(Point pt){
        if(this.period == null) return;

        RowsLtoR row = hittedRow(pt);
        if(row == null) return;
        if( ! (row instanceof TalkDraw) ) return;
        
        TalkDraw chat = (TalkDraw) row;
        
        int index = chat.getRegexMatchIndex(pt);
        if(index < 0) return;
        
        clearHotTarget();
        chat.setHotTargetIndex(index);
        
        return;
    }
    
    /**
     * マウスリスナー。
     * アンカーヒット処理を行う。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseClicked(MouseEvent ev){
        Point pt = ev.getPoint();
        if(ev.getButton() == MouseEvent.BUTTON1){
            clearSelect();
            hitAnchor(pt);
            hitRegex(pt);
        }
        return;
    }

    /**
     * マウスリスナー。
     * 何もしない。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseEntered(MouseEvent ev){
        return;
    }

    /**
     * マウスリスナー。
     * 何もしない。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseExited(MouseEvent ev){
        return;
    }

    /**
     * マウスリスナー。
     * ドラッグ開始処理を行う。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mousePressed(MouseEvent ev){
        requestFocusInWindow();
        
        this.lastPressedPoint = ev.getPoint();
        
        if(ev.getButton() == MouseEvent.BUTTON1){
            clearSelect();
            this.dragFrom = ev.getPoint();
        }
        
        return;
    }

    /**
     * マウスリスナー。
     * ドラッグ終了処理を行う。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseReleased(MouseEvent ev){
        if(ev.getButton() == MouseEvent.BUTTON1){
            this.dragFrom = null;
        }
        return;
    }

    /**
     * マウスリスナー。
     * ドラッグ処理を行う。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseDragged(MouseEvent ev){
        if(this.dragFrom == null) return;
        Point dragTo = ev.getPoint();
        drag(this.dragFrom, dragTo);
        return;
    }

    /**
     * マウスリスナー。
     * 何もしない。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseMoved(MouseEvent ev){
        return;
    }
    
    /**
     * 最後にマウスボタンが押された場所を返す。
     * @return マウス座標。
     */
    public Point getLastPressedPoint(){
        return this.lastPressedPoint;
    }
    
    /**
     * 選択文字列を返す。
     * @return 選択文字列
     */
    public CharSequence getSelected(){
        StringBuilder selected = new StringBuilder();
        
        for(RowsLtoR row : this.rowList){
            if(isFiltered(row)) continue;
            try{
                row.appendSelected(selected);
            }catch(IOException e){
                assert false; // ありえない
                return null;
            }
        }
        
        if(selected.length() <= 0) return null;
        
        return selected;
    }
    
    /**
     * 文字列をクリップボードにコピーする。
     * @param data 文字列
     */
    private void copyToClipBoard(CharSequence data){
        Toolkit toolkit = Toolkit.getDefaultToolkit();
        Clipboard clipboard = toolkit.getSystemClipboard();
        StringSelection selection = new StringSelection(data.toString());
        clipboard.setContents(selection, selection);
        return;
    }
    
    /**
     * 選択文字列をクリップボードにコピーする。
     * @return 選択文字列
     */
    public CharSequence copySelected(){
        CharSequence selected = getSelected();
        if(selected == null) return null;
        copyToClipBoard(selected);
        return selected;
    }

    /**
     * 矩形の示す一発言をクリップボードにコピーする。
     * @return コピーした文字列
     */
    public CharSequence copyTalk(){
        RowsLtoR row = this.popup.lastPopupedRow;
        if(row == null) return null;
        if( ! (row instanceof TalkDraw) ) return null;
        TalkDraw talkDraw = (TalkDraw) row;
        Talk talk = talkDraw.getTalk();
        
        StringBuilder selected = new StringBuilder();

        Avatar avatar = talk.getAvatar();
        selected.append(avatar.getName()).append(' ');
        
        String anchor = talk.getAnchorNotation();
        selected.append(anchor).append('\n');
        
        selected.append(talk.getDialog());
        if(selected.charAt(selected.length()-1) != '\n'){
            selected.append('\n');
        }
        
        copyToClipBoard(selected);
        
        return selected;
    }
    
    /**
     * Look&Feelの更新
     */
    @Override
    public void updateUI(){
        super.updateUI();
        this.popup.updateUI();
        
        updateInputMap();
        
        return;
    }

    /**
     * COPY処理を行うキーの設定をJTextFieldから流用する。
     */
    private void updateInputMap(){
        InputMap thisInputMap = getInputMap();

        InputMap sampleInputMap;
        sampleInputMap = new JTextField().getInputMap();
        KeyStroke[] strokes = sampleInputMap.allKeys();
        for(KeyStroke stroke : strokes){
            Object bind = sampleInputMap.get(stroke);
            if(bind.equals(DefaultEditorKit.copyAction)){
                thisInputMap.put(stroke, DefaultEditorKit.copyAction);
            }
        }

        return;
    }

    /**
     * ActionListenerを追加する。
     * @param listener リスナー
     */
    public void addActionListener(ActionListener listener){
        this.thisListenerList.add(ActionListener.class, listener);

        this.popup.menuCopy   .addActionListener(listener);
        this.popup.menuSelTalk.addActionListener(listener);
        this.popup.menuSummary.addActionListener(listener);
            
        return;
    }
    
    /**
     * ActionListenerを削除する。
     * @param listener リスナー
     */
    public void removeActionListener(ActionListener listener){
        this.thisListenerList.remove(ActionListener.class, listener);

        this.popup.menuCopy   .removeActionListener(listener);
        this.popup.menuSelTalk.removeActionListener(listener);
        this.popup.menuSummary.removeActionListener(listener);

        return;
    }

    /**
     * ActionListenerを列挙する。
     * @return すべてのActionListener
     */
    public ActionListener[] getActionListeners(){
        return this.thisListenerList.getListeners(ActionListener.class);
    }
    
    /**
     * AnchorHitListenerを追加する。
     * @param listener リスナー
     */
    public void addAnchorHitListener(AnchorHitListener listener){
        this.thisListenerList.add(AnchorHitListener.class, listener);
        return;
    }
    
    /**
     * AnchorHitListenerを削除する。
     * @param listener リスナー
     */
    public void removeAnchorHitListener(AnchorHitListener listener){
        this.thisListenerList.remove(AnchorHitListener.class, listener);
        return;
    }

    /**
     * AnchorHitListenerを列挙する。
     * @return すべてのAnchorHitListener
     */
    public AnchorHitListener[] getAnchorHitListeners(){
        return this.thisListenerList.getListeners(AnchorHitListener.class);
    }
    
    /**
     * イベントリスナーを列挙する
     * @param <T>
     * @param listenerType イベントリスナーの型
     * @return 指定した型のイベントリスナーすべて
     */
    @Override
    public <T extends EventListener> T[] getListeners(Class<T> listenerType){
	T[] result;
        result = this.thisListenerList.getListeners(listenerType); 
        
        if(result.length <= 0){ 
	    result = super.getListeners(listenerType); 
	}

        return result;
    }
    
    /**
     * キーボード入力用ダミーAction
     */
    private class ProxyAction extends AbstractAction{
        
        private String command;
        
        public ProxyAction(String command){
            this.command = command;
            return;
        }
        
        public void actionPerformed(ActionEvent event){
            Object source  = event.getSource();
            int id         = event.getID();
            String actcmd  = this.command;
            long when      = event.getWhen();
            int modifiers  = event.getModifiers();
            
            for(ActionListener listener : getActionListeners()){
                ActionEvent newEvent = new ActionEvent(source,
                                                       id,
                                                       actcmd,
                                                       when,
                                                       modifiers );
                listener.actionPerformed(newEvent);
            }
            return;
        }
    };
    
    /**
     * ポップアップメニュー
     */
    private class DiscussionPopup extends JPopupMenu{
        
        private final JMenuItem menuCopy = new JMenuItem("選択範囲をコピー");
        private final JMenuItem menuSelTalk = new JMenuItem("この発言をコピー");
        private final JMenuItem menuSummary = new JMenuItem("発言を集計...");
        private RowsLtoR lastPopupedRow;
        
        /**
         * コンストラクタ
         */
        public DiscussionPopup(){
            super();
            
            add(this.menuCopy);
            add(this.menuSelTalk);
            addSeparator();
            add(this.menuSummary);
            
            this.menuCopy   .setActionCommand(ActionManager.COMMAND_COPY);
            this.menuSelTalk.setActionCommand(ActionManager.COMMAND_COPYTALK);
            this.menuSummary.setActionCommand(ActionManager.COMMAND_DAYSUMMARY);
            
            return;
        }

        /**
         * ポップアップの表示時に呼ばれる。
         * @param comp 原因となったコンポーネント
         * @param x マウスのX座標
         * @param y マウスのY座標
         */
        @Override
        public void show(Component comp, int x, int y){
            this.lastPopupedRow = hittedRow(new Point(x,y));
            if(   this.lastPopupedRow != null
               && this.lastPopupedRow instanceof TalkDraw){
                this.menuSelTalk.setEnabled(true);
            }else{
                this.menuSelTalk.setEnabled(false);
            }
            
            if(Discussion.this.getSelected() != null){
                this.menuCopy.setEnabled(true);
            }else{
                this.menuCopy.setEnabled(false);
            }
                
            super.show(comp, x, y);
            
            return;
        }
    }
}
