/*
 * Regex pattern
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: RegexPattern.java 216 2008-09-30 09:35:29Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * 正規表現
 */
public class RegexPattern{
    private static final String REGEX_ITEM_PREFIX = "[R] ";

    private final String editSource;
    private final boolean isRegex;
    private final Pattern pattern;
    private final int flag;
    private final String comment;
    
    /**
     * コンストラクタ。
     * 
     * @param editSource リテラル文字列または正規表現
     * @param isRegex 指定文字列が正規表現ならtrue。リテラルならfalse
     * @param flag 正規表現フラグ
     * @param comment コメント
     * @throws java.util.regex.PatternSyntaxException 正規表現がおかしい
     */
    public RegexPattern(String editSource,
                          boolean isRegex,
                          int flag,
                          String comment)
            throws PatternSyntaxException{
        super();

        this.editSource = editSource;
        this.isRegex    = isRegex;
        this.flag       = flag;
        this.comment    = comment;

        String regexExpr;
        if(this.isRegex){
            regexExpr = this.editSource;
        }else{
            regexExpr = Pattern.quote(this.editSource);
        }

        this.pattern = Pattern.compile(regexExpr, this.flag);

        return;
    }

    /**
     * コンストラクタ。
     * 
     * @param editSource リテラル文字列または正規表現
     * @param isRegex 指定文字列が正規表現ならtrue。リテラルならfalse
     * @param flag 正規表現フラグ
     * @throws java.util.regex.PatternSyntaxException 正規表現がおかしい
     */
    public RegexPattern(String editSource,
                          boolean isRegex,
                          int flag)
                          throws PatternSyntaxException{
        this(editSource, isRegex, flag, "");
        return;
    }
    
    /**
     * 元の入力文字列を返す。
     * @return 入力文字列
     */
    public String getEditSource(){
        return this.editSource;
    }

    /**
     * 元の入力文字列が正規表現か否か返す。
     * @return 正規表現ならtrue
     */
    public boolean isRegex(){
        return this.isRegex;
    }

    /**
     * 正規表現フラグを返す。
     * @return 正規表現フラグ。
     */
    public int getFlag(){
        return this.flag;
    }

    /**
     * コンパイルされた正規表現形式を返す。
     * @return コンパイルされた正規表現形式
     */
    public Pattern getPattern(){
        return this.pattern;
    }

    /**
     * リストアイテムにふさわしい文字列表記を返す。
     * @return 文字列表記
     */
    public String getListItemString(){
        String result;
        if(isRegex()){
            result = REGEX_ITEM_PREFIX + getEditSource();
        }else{
            result = getEditSource();
        }
        result += this.comment;
        return result;
    }

    /**
     * 文字列表記を返す。
     * @return 文字列表記
     */
    @Override
    public String toString(){
        return getListItemString();
    }
}
