/*
 * CharSequence CharacterIterator
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: SequenceCharacterIterator.java 154 2008-09-04 16:15:41Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.text.CharacterIterator;

/**
 * CharSequenceをソースとするCharacterIterator
 */
public class SequenceCharacterIterator implements CharacterIterator{
    
    private final CharSequence source;
    private int cursorBegin;
    private int cursorEnd;
    private int cursorLength;    
    private int cursorPos;

    /**
     * コンストラクタ
     * @param source ソース文字列
     * @param cursorBegin カーソル開始位置
     * @param cursorEnd カーソル終了位置
     */
    public SequenceCharacterIterator(CharSequence source,
                                         int cursorBegin, int cursorEnd){
        super();
        
        if(cursorBegin > cursorEnd){
            throw new IllegalArgumentException();
        }
        if(cursorBegin < 0 || source.length() < cursorEnd){
            throw new IndexOutOfBoundsException();
        }

        this.source = source;
        this.cursorBegin = cursorBegin;
        this.cursorEnd   = cursorEnd;
        this.cursorLength = this.cursorEnd - this.cursorBegin;
        this.cursorPos = this.cursorBegin;

        return;
    }
    
    /**
     * コンストラクタ
     * @param source ソース文字列
     */
    public SequenceCharacterIterator(CharSequence source){
        super();
        this.source = source;
        this.cursorBegin = 0;
        this.cursorEnd   = source.length();
        this.cursorLength = this.cursorEnd - this.cursorBegin;
        this.cursorPos = this.cursorBegin;
        return;
    }

    /**
     * カーソル位置を先頭に設定する。
     * @return カーソル位置の示す文字
     */
    public char first(){
        this.cursorPos = this.cursorBegin;
        return current();
    }

    /**
     * カーソル位置を最後に設定する。
     * @return カーソル位置の示す文字
     */
    public char last(){
        this.cursorPos = this.cursorEnd - 1;
        return current();
    }

    /**
     * カーソル位置の示す文字を返す。
     * @return カーソル位置の示す文字
     */
    public char current(){
        if(this.cursorLength <= 0 || this.cursorPos < this.cursorBegin){
            this.cursorPos = this.cursorBegin;
            return CharacterIterator.DONE;
        }
        if(this.cursorPos >= this.cursorEnd){
            this.cursorPos = this.cursorEnd;
            return CharacterIterator.DONE;
        }
        return this.source.charAt(this.cursorPos);
    }

    /**
     * カーソルを後ろに移動する。
     * @return 移動後のカーソル位置の示す文字
     */
    public char next(){
        this.cursorPos++;
        return current();
    }

    /**
     * カーソルを前に移動する。
     * @return 移動後のカーソル位置の示す文字
     */
    public char previous(){
        this.cursorPos--;
        return current();
    }

    /**
     * カーソル位置を設定する。
     * @param newPos 新しいカーソル位置
     * @return カーソル位置の示す文字
     * @throws java.lang.IllegalArgumentException カーソル位置が変
     */
    public char setIndex(int newPos) throws IllegalArgumentException{
        if(newPos < this.cursorBegin || this.cursorEnd < newPos){
            throw new IllegalArgumentException();
        }
        this.cursorPos = newPos;
        return current();
    }

    /**
     * カーソル開始位置を返す。
     * @return カーソル開始位置。
     */
    public int getBeginIndex(){
        return this.cursorBegin;
    }

    /**
     * カーソル終了位置を返す。
     * @return カーソル終了位置。
     */
    public int getEndIndex(){
        return this.cursorEnd;
    }

    /**
     * カーソル位置を返す。
     * @return カーソル位置
     */
    public int getIndex(){
        return this.cursorPos;
    }

    /**
     * シャローコピーを返す。
     * @return コピー
     */
    @Override
    public Object clone(){
        Object result;
        try{
            result = super.clone();
        }catch(CloneNotSupportedException e){
            assert false;
            return null;
        }
        return result;
    }
}
