/*
 * discussion browser
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Discussion.java 157 2008-09-04 17:37:06Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.font.FontRenderContext;
import java.awt.geom.AffineTransform;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.Scrollable;
import javax.swing.SwingConstants;
import javax.swing.event.MouseInputListener;
import javax.swing.text.DefaultEditorKit;

/**
 * 発言表示画面
 */
@SuppressWarnings("serial")
public class Discussion extends JComponent
        implements Scrollable, MouseInputListener{

    private static final Insets insets = new Insets(15, 15, 15, 15);
    private static final int horizontalGap = insets.left + insets.right;
    private static final int verticalGap = insets.top + insets.bottom;
    
    private static final Font font = new Font("Dialog", Font.PLAIN, 20);
    private static final boolean antiAliaseText;
    private static final boolean useFractional;
    private static final FontRenderContext frc;
    private static final RenderingHints hints = new RenderingHints(null);
    
    static{
        if(   AppSetting.osName     != null
           && AppSetting.javaVendor != null
           && AppSetting.osName    .toUpperCase().contains("WINDOWS")
           && AppSetting.javaVendor.toUpperCase().contains("SUN")    ){
            antiAliaseText = false;    // Win+SunJRE 向け
            useFractional  = false;
        }else{
            antiAliaseText = true;     // Mac OS X を含むその他向け
            useFractional  = true;
        }
        
        hints.put(RenderingHints.KEY_ANTIALIASING,
                  RenderingHints.VALUE_ANTIALIAS_ON);
        
        if(antiAliaseText){
            hints.put(RenderingHints.KEY_TEXT_ANTIALIASING,
                  RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
        }else{
            hints.put(RenderingHints.KEY_TEXT_ANTIALIASING,
                  RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
        }
        
        if(useFractional){
            hints.put(RenderingHints.KEY_FRACTIONALMETRICS,
                      RenderingHints.VALUE_FRACTIONALMETRICS_ON);
        }else{
            hints.put(RenderingHints.KEY_FRACTIONALMETRICS,
                      RenderingHints.VALUE_FRACTIONALMETRICS_OFF);
        }

        AffineTransform affineTx = new AffineTransform();
        frc = new FontRenderContext(affineTx, antiAliaseText, useFractional);
    }

    private Period period;
    private TopicFilter topicFilter = null;
    private TopicFilter.FilterContext filterContext;
    private Pattern searchRegex = null;
    private final List<RowsLtoR> rowList = new LinkedList<RowsLtoR>();
    private Dimension idealSize;
    private int lastWidth = -1;
    private Point dragFrom;
    private DiscussionPopup popup = new DiscussionPopup();
    
    private Action copySelectedAction = new AbstractAction(){
        public void actionPerformed(ActionEvent event){
            Discussion.this.copySelected();
            return;
        }
    };
    
    /**
     * 初期Periodを指定してメインブラウザを作成する。
     * @param period 初期Period
     */
    public Discussion(Period period){
        super();

        setPeriod(period);

        addMouseListener(this);
        addMouseMotionListener(this);

        setComponentPopupMenu(this.popup);

        updateInputMap();
        ActionMap actionMap = getActionMap();
        actionMap.put(DefaultEditorKit.copyAction, this.copySelectedAction);

        return;
    }

    /**
     * 現在のPeriodを返す。
     * @return 現在のPeriod
     */
    public Period getPeriod(){
        return this.period;
    }

    /**
     * Periodを更新する。
     * 古いPeriodの表示内容は消える。
     * 新しいPeriodの表示内容はまだ反映されない。
     * @param period 新しいPeriod
     */
    public void setPeriod(Period period){
        if(period == null){
            this.period = null;
            this.rowList.clear();
            return;
        }
        
        if(   this.period == period
           && period.getTopics() == this.rowList.size() ){
            return;
        }

        this.period = period;

        this.filterContext = null;

        this.rowList.clear();
        for(Topic topic : this.period.getTopicList()){
            RowsLtoR row;
            if(topic instanceof Talk){
                Talk talk = (Talk) topic;
                row = new TalkDraw(talk, font, frc);
            }else if(topic instanceof SysEvent){
                SysEvent sysEvent = (SysEvent) topic;
                row = new SysEventDraw(sysEvent, font, frc);
            }else{
                assert false;
                continue;
            }
            this.rowList.add(row);
        }

        clearSizeCache();
        int width = getWidth();
        calcIdealSize(width);
        
        repaint();
        revalidate();
        
        return;
    }

    /**
     * インセットを返す。
     * @return インセット
     */
    @Override
    public Insets getInsets(){
        return insets;
    }
    
    /**
     * フィルタを適用してPeriodの内容を出力する。
     * @param force trueなら強制再出力
     */
    public void showTopics(boolean force){
        setPeriod(getPeriod()); // TODO Periodからのイベントを受け取るべし？
        return;
    }

    /**
     * 発言フィルタを設定する。
     * @param filter
     */
    public void setTopicFilter(TopicFilter filter){
        this.topicFilter = filter;
        filtering();
        return;
    }

    /**
     * 発言フィルタを適用する。
     */
    public void filtering(){
        if(   this.topicFilter != null
           && this.topicFilter.isSame(this.filterContext)){
            return;
        }

        if(this.topicFilter != null){
            this.filterContext = this.topicFilter.getFilterContext();
        }else{
            this.filterContext = null;
        }

        int width = getWidth();
        calcIdealSizeIntl(width);

        repaint();
        revalidate();
        
        return;
    }

    /**
     * 検索パターンを設定する。
     * @param searchRegex 検索パターン
     */
    public void setSearchRegex(Pattern searchRegex){
        if(this.searchRegex == searchRegex) return;
        this.searchRegex = searchRegex;
        return;
    }

    /**
     * 与えられた正規表現にマッチする文字列をハイライト描画する。
     * @return ヒット件数
     */
    public int highlightRegex(){
        int total = 0;
        
        for(RowsLtoR row : this.rowList){
            if( ! (row instanceof TalkDraw) ) continue;
            TalkDraw talkDraw = (TalkDraw) row;
            total += talkDraw.setRegex(this.searchRegex);
        }
        
        repaint();
        
        return total;
    }
    
    /**
     * 過去に計算した寸法を破棄する。
     */
    private void clearSizeCache(){
        this.idealSize = null;
        this.lastWidth = -1;
        revalidate();
    }
    
    /**
     * 指定した矩形がフィルタリング対象か判定する。
     * @param row 矩形
     * @return フィルタリング対象ならtrue
     */
    private boolean isFiltered(RowsLtoR row){
        if(this.topicFilter == null) return false;
        
        Topic topic;
        if(row instanceof TalkDraw){
            topic = ((TalkDraw)row).getTalk();
        }else if(row instanceof SysEventDraw){
            topic = ((SysEventDraw)row).getSysEvent();
        }else{
            return false;
        }
        
        return this.topicFilter.isFiltered(topic);
    }
    
    /**
     * 指定位置から指定幅で描画した場合の寸法を返す。
     * @param xPos 指定位置X座標
     * @param yPos 指定位置Y座標
     * @param widthLimit 指定幅ピクセル数
     * @return 寸法
     */
    private Rectangle calcRect(int xPos, int yPos, int widthLimit){
        final int innerLimit = widthLimit - horizontalGap;
        Rectangle unionRect = null;
        for(RowsLtoR row : this.rowList){

            if(isFiltered(row)) continue;
            
            row.setPos(xPos, yPos);
            Rectangle rowBound = row.setWidth(innerLimit);
            yPos += rowBound.height;
            
            if(unionRect == null){
                unionRect = new Rectangle(rowBound);
            }else{
                unionRect.add(rowBound);
            }
        }
        return unionRect;
    }
    
    /**
     * 指定された幅を満たす寸法でPreferredSizeを更新する。
     * @param newWidth 指定幅ピクセル数
     */
    private void calcIdealSize(final int newWidth){
        if(newWidth == this.lastWidth) return;
        calcIdealSizeIntl(newWidth);
    }
    
    /**
     * 指定された幅を満たす寸法でPreferredSizeを更新する。
     * @param newWidth 指定幅ピクセル数
     */
    private void calcIdealSizeIntl(final int newWidth){
        this.lastWidth = newWidth;

        Rectangle unionRect = calcRect(insets.left,
                                       insets.top,
                                       newWidth );
        
        if(this.idealSize == null){
            this.idealSize = new Dimension();
        }

        int newHeight = verticalGap;
        if(unionRect != null){
            newHeight += unionRect.height;
        }

        this.idealSize.width  = newWidth;
        this.idealSize.height = newHeight;
        
        setPreferredSize(this.idealSize);
        
        return;
    }
    
    /**
     * コンポーネント内を描画。
     * @param g グラフィックコンテキスト
     */
    @Override
    public void paintComponent(Graphics g){
        Graphics2D g2 = (Graphics2D) g;
        g2.setRenderingHints(hints);
        
        g2.setColor(Color.BLACK);
        Rectangle clipRect = g.getClipBounds();
        g2.fillRect(clipRect.x, clipRect.y, clipRect.width, clipRect.height);

        for(RowsLtoR row : this.rowList){
            Rectangle rowRect = row.getBounds();
            if( ! rowRect.intersects(clipRect) ) continue;
            
            if(isFiltered(row)) continue;
            
            row.paint(g2);
        }
        
        return;
    }

    /**
     * コンポーネントのサイズを設定する。
     * @param size
     */
    @Override
    public void setSize(Dimension size){
        setBounds(getX(), getY(), size.width, size.height);
        return;
    }
    
    /**
     * コンポーネントのサイズを設定する。
     * @param width
     * @param height
     */
    @Override
    public void setSize(int width, int height){
        setBounds(getX(), getY(), width, height);
        return;
    }
    
    /**
     * コンポーネントのサイズを設定する。
     * @param size
     * @deprecated
     */
    @Override
    @Deprecated
    public void resize(Dimension size){
        setBounds(getX(), getY(), size.width, size.height);
        return;
    }
    
    /**
     * コンポーネントのサイズを設定する。
     * @param width
     * @param height
     * @deprecated
     */
    @Override
    @Deprecated
    public void resize(int width, int height){
        setBounds(getX(), getY(), width, height);
        return;
    }

    /**
     * コンポーネントのサイズと位置を設定する。
     * @param rect
     */
    @Override
    public void setBounds(Rectangle rect){
        setBounds(rect.x, rect.y, rect.width, rect.height);
    }
    
    /**
     * コンポーネントのサイズと位置を設定する。
     * @param x
     * @param y
     * @param width
     * @param height
     */
    @Override
    public void setBounds(int x, int y, int width, int height){
        int oldWidth = getWidth();
        super.setBounds(x, y, width, height);
        if(oldWidth == width) return;

        calcIdealSize(width);
        if(   this.idealSize.width != width
           || this.idealSize.height != height ){
            revalidate();
        }
        
        return;
    }
    
    /**
     * ビューポートの推奨サイズを返す。
     * @return 推奨サイズ
     */
    public Dimension getPreferredScrollableViewportSize(){
        return getPreferredSize();
    }

    /**
     * ※ Scrollable参照
     * @return ビューポートの幅に合わせるならtrue
     */
    public boolean getScrollableTracksViewportWidth(){
        return true;
    }

    /**
     * ※ Scrollable参照
     * @return ビューポートの高さに合わせるならtrue
     */
    public boolean getScrollableTracksViewportHeight(){
        return false;
    }

    /**
     * ブロック単位のスクロール量を返す。
     * @param visibleRect ビューポート内可視領域
     * @param orientation スクロール方向
     * @param direction スクロール向き
     * @return スクロール量
     */
    public int getScrollableBlockIncrement(Rectangle visibleRect,
                                               int orientation,
                                               int direction        ){
        if(orientation == SwingConstants.VERTICAL){
            return visibleRect.height;
        }
        return 30; // TODO フォント高 × 1.5 ぐらい？
    }

    /**
     * 行単位のスクロール量を返す。
     * @param visibleRect ビューポート内可視領域
     * @param orientation スクロール方向
     * @param direction スクロール向き
     * @return スクロール量
     */
    public int getScrollableUnitIncrement(Rectangle visibleRect,
                                              int orientation,
                                              int direction      ){
        return 30;
    }
    
    /**
     * ドラッグ処理を行う。
     * @param from ドラッグ開始位置
     * @param to 現在のドラッグ位置
     */
    private void drag(Point from, Point to){
        Rectangle dragRegion = new Rectangle();
        dragRegion.setFrameFromDiagonal(from, to);
        
        for(RowsLtoR row : this.rowList){
            if( ! row.getBounds().intersects(dragRegion) ) continue;
            row.drag(from, to);
        }
        repaint();
        return;
    }
    
    /**
     * 選択範囲の解除。
     */
    private void clearSelect(){
        for(RowsLtoR row : this.rowList){
            row.clearSelect();
        }
        repaint();
        return;
    }
    
    /**
     * 与えられた点座標を包含する矩形を返す。
     * @param pt 点座標（JComponent基準）
     * @return 点座標を含む矩形。含む矩形がなければnullを返す。
     */
    private RowsLtoR hitCheck(Point pt){
        for(RowsLtoR row : this.rowList){
            Rectangle bounds = row.getBounds();
            if(bounds.contains(pt)) return row;
        }
        return null;
    }
    
    /**
     * アンカークリック動作の処理。
     * @param pt クリックポイント
     */
    // TODO Controllerへ移したい。
    private void hitAnchor(Point pt){
        if(this.period == null) return;

        RowsLtoR row = hitCheck(pt);
        if(row == null) return;
        if( ! (row instanceof TalkDraw) ) return;
        
        TalkDraw chat = (TalkDraw) row;
        Anchor anchor = chat.getAnchor(pt);
        if(anchor == null) return;

        Village village = this.period.getVillage();
        List<Talk> talkList;
        try{
            talkList = village.getTalkListFromAnchor(anchor);
        }catch(IOException e){
            return;
        }
        if(talkList.size() <= 0) return;
        
        chat.showAnchorTalks(anchor, talkList);

        int width = getWidth();
        calcIdealSizeIntl(width);

        repaint();
        revalidate();
        
        return;
    }
    
    /**
     * マウスリスナー。
     * アンカーヒット処理を行う。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseClicked(MouseEvent ev){
        Point pt = ev.getPoint();
        if(ev.getButton() == MouseEvent.BUTTON1){
            clearSelect();
            hitAnchor(pt);
        }
        return;
    }

    /**
     * マウスリスナー。
     * 何もしない。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseEntered(MouseEvent ev){
        return;
    }

    /**
     * マウスリスナー。
     * 何もしない。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseExited(MouseEvent ev){
        return;
    }

    /**
     * マウスリスナー。
     * ドラッグ開始処理を行う。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mousePressed(MouseEvent ev){
        requestFocusInWindow();
        
        if(ev.getButton() == MouseEvent.BUTTON1){
            clearSelect();
            this.dragFrom = ev.getPoint();
        }
        
        return;
    }

    /**
     * マウスリスナー。
     * ドラッグ終了処理を行う。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseReleased(MouseEvent ev){
        if(ev.getButton() == MouseEvent.BUTTON1){
            this.dragFrom = null;
        }
        return;
    }

    /**
     * マウスリスナー。
     * ドラッグ処理を行う。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseDragged(MouseEvent ev){
        if(this.dragFrom == null) return;
        Point dragTo = ev.getPoint();
        drag(this.dragFrom, dragTo);
        return;
    }

    /**
     * マウスリスナー。
     * 何もしない。
     * MouseInputListenerを参照せよ。
     * @param ev マウスイベント
     */
    public void mouseMoved(MouseEvent ev){
        return;
    }
    
    /**
     * 選択文字列を返す。
     * @return 選択文字列
     */
    public CharSequence getSelected(){
        StringBuilder selected = new StringBuilder();
        
        for(RowsLtoR row : this.rowList){
            try{
                row.appendSelected(selected);
            }catch(IOException e){
                assert false; // ありえない
                return null;
            }
        }
        
        if(selected.length() <= 0) return null;
        
        return selected;
    }
    
    /**
     * 文字列をクリップボードにコピーする。
     * @param data 文字列
     */
    private void copyToClipBoard(CharSequence data){
        Toolkit toolkit = Toolkit.getDefaultToolkit();
        Clipboard clipboard = toolkit.getSystemClipboard();
        StringSelection selection = new StringSelection(data.toString());
        clipboard.setContents(selection, selection);
        return;
    }
    
    /**
     * 選択文字列をクリップボードにコピーする。
     * @return 選択文字列
     */
    public CharSequence copySelected(){
        CharSequence selected = getSelected();
        if(selected == null) return null;
        copyToClipBoard(selected);
        return selected;
    }

    /**
     * 矩形の示す一発言をクリップボードにコピーする。
     * @param row 矩形
     * @return コピーした文字列
     */
    public CharSequence copyTalk(RowsLtoR row){
        if(row == null) return null;
        if( ! (row instanceof TalkDraw) ) return null;
        TalkDraw talkDraw = (TalkDraw) row;
        Talk talk = talkDraw.getTalk();
        
        StringBuilder selected = new StringBuilder();

        Avatar avatar = talk.getAvatar();
        selected.append(avatar.getName()).append(' ');
        
        String anchor = talk.getAnchorNotation();
        selected.append(anchor).append('\n');
        
        selected.append(talk.getDialog());
        if(selected.charAt(selected.length()-1) != '\n'){
            selected.append('\n');
        }
        
        copyToClipBoard(selected);
        
        return selected;
    }
    
    /**
     * Look&Feelの更新
     */
    @Override
    public void updateUI(){
        super.updateUI();
        updateInputMap();
        return;
    }
    
    /**
     * COPY処理を行うキーの設定をJTextFieldから流用する。
     */
    private void updateInputMap(){
        InputMap thisInputMap = getInputMap();

        InputMap sampleInputMap;
        sampleInputMap = new JTextField().getInputMap();
        KeyStroke[] strokes = sampleInputMap.allKeys();
        for(KeyStroke stroke : strokes){
            Object bind = sampleInputMap.get(stroke);
            if(bind.equals(DefaultEditorKit.copyAction)){
                thisInputMap.put(stroke, DefaultEditorKit.copyAction);
            }
        }

        return;
    }

    /**
     * ポップアップメニュー
     */
    private class DiscussionPopup extends JPopupMenu
            implements ActionListener{
        private final JMenuItem menuCopy = new JMenuItem("選択範囲をコピー");
        private final JMenuItem menuSelTalk = new JMenuItem("この発言をコピー");
        private RowsLtoR lastPopupedRow;
        
        /**
         * コンストラクタ
         */
        public DiscussionPopup(){
            super();
            
            add(this.menuCopy);
            add(this.menuSelTalk);
            
            this.menuCopy.setActionCommand(MenuManager.COMMAND_COPY);
            this.menuSelTalk.setActionCommand(MenuManager.COMMAND_COPYTALK);
            
            this.menuCopy   .addActionListener(this);
            this.menuSelTalk.addActionListener(this);
            
            return;
        }

        /**
         * ポップアップの表示時に呼ばれる。
         * @param comp 原因となったコンポーネント
         * @param x マウスのX座標
         * @param y マウスのY座標
         */
        @Override
        public void show(Component comp, int x, int y){
            this.lastPopupedRow = hitCheck(new Point(x,y));
            if(   this.lastPopupedRow != null
               && this.lastPopupedRow instanceof TalkDraw){
                this.menuSelTalk.setEnabled(true);
            }else{
                this.menuSelTalk.setEnabled(false);
            }
            
            if(Discussion.this.getSelected() != null){
                this.menuCopy.setEnabled(true);
            }else{
                this.menuCopy.setEnabled(false);
            }
                
            super.show(comp, x, y);
            
            return;
        }

        /**
         * ポップアップメニュー選択時の処理。
         * @param event Actionイベント
         */
        public void actionPerformed(ActionEvent event){
            String cmd = event.getActionCommand();
            if(cmd.equals(MenuManager.COMMAND_COPY)){
                Discussion.this.copySelected();
            }else if(cmd.equals(MenuManager.COMMAND_COPYTALK)){
                Discussion.this.copyTalk(this.lastPopupedRow);
            }

            return;
        }
    }
}
