/*
 * application settings
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: AppSetting.java 114 2008-07-17 10:47:40Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.io.File;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * アプリケーション及び実行環境の各種設定
 */
public class AppSetting{
    
    private static final Map<String, String> propertyMap =
            new TreeMap<String, String>();
    public static final String osName;
    public static final String osVersion;
    public static final String osArch;
    public static final String javaVendor;
    public static final String javaVersion;
    public static final String classpath;
    
    public static final String proxyHost;
    public static final String proxyPort;
    public static final String nonProxyHosts;
    
    private static final Map<String, String> environmentMap =
            new TreeMap<String, String>();
    public static final String envLANG;
//  public static final String envPATH;
    public static final String envDISPLAY;
    
    public static final String[] classpaths;
//  public static final String[] paths;
    
    public static boolean needHelp = false;
    public static boolean needVersion = false;
    public static boolean needVMInfo = false;
    public static boolean boldMetal = false;
    public static boolean noSplash = false;
    public static boolean consolelog = false;
    
    public static int frameWidth = 800;
    public static int frameHeight = 600;
    public static int frameXpos = Integer.MIN_VALUE;
    public static int frameYpos = Integer.MIN_VALUE;
    
    private static final List<String> invokeArgs = new LinkedList<String>();

    private static final Pattern geomPattern =
            Pattern.compile(
                 "([1-9][0-9]*)x([1-9][0-9]*)"
                +"(?:(\\+|\\-)([1-9][0-9]*)(\\+|\\-)([1-9][0-9]*))?"
                );

    static{
        osName      = getSecureProperty("os.name");
        osVersion   = getSecureProperty("os.version");
        osArch      = getSecureProperty("os.arch");
        javaVendor  = getSecureProperty("java.vendor");
        javaVersion = getSecureProperty("java.version");
        classpath   = getSecureProperty("java.class.path");

        proxyHost     = getSecureProperty("http.proxyHost");
        proxyPort     = getSecureProperty("http.proxyPort");
        nonProxyHosts = getSecureProperty("http.nonProxyHosts");

        envLANG      = getSecureEnvironment("LANG");
//      envPATH      = getSecureEnvironment("PATH");
        envDISPLAY   = getSecureEnvironment("DISPLAY");
        
        if(classpath != null){
            classpaths = classpath.split(File.pathSeparator);
        }else{
            classpaths = new String[0];
        }
/*        
        if(envPATH != null){
            paths = envPATH.split(File.pathSeparator);
        }else{
            paths = new String[0];
        }
*/
    }

    /**
     * 可能ならシステムプロパティを読み込む。
     * @param key キー
     * @return プロパティ値。セキュリティ上読み込み禁止の場合はnull。
     */
    private static String getSecureProperty(String key){
        String result;
        try{
            result = System.getProperty(key);
            if(result != null) propertyMap.put(key, result);
        }catch(SecurityException e){
            result = null;
        }
        return result;
    }
    
    /**
     * 可能なら環境変数を読み込む。
     * @param name 環境変数名
     * @return 環境変数値。セキュリティ上読み込み禁止の場合はnull。
     */
    private static String getSecureEnvironment(String name){
        String result;
        try{
            result = System.getenv(name);
            if(result != null) environmentMap.put(name, result);
        }catch(SecurityException e){
            result = null;
        }
        return result;
    }
    
    /**
     * VM詳細情報を文字列化する。
     * @return VM詳細情報
     */
    public static String getVMInfo(){
        StringBuilder result = new StringBuilder();

        result.append("最大ヒープメモリ量: "
                + Jindolf.runtime.maxMemory() + " bytes\n");

        result.append("\n");
        
        result.append("起動時引数:\n");
        for(String arg : invokeArgs){
            result.append("  ").append(arg).append("\n");
        }
        
        result.append("\n");

        result.append("主要システムプロパティ:\n");
        Set<String> propKeys = propertyMap.keySet();
        for(String propKey : propKeys){
            if(propKey.equals("java.class.path")) continue;
            String value = propertyMap.get(propKey);
            result.append("  ");
            result.append(propKey).append("=").append(value).append("\n");
        }
        
        result.append("\n");
        
        result.append("主要環境変数:\n");
        Set<String> envKeys = environmentMap.keySet();
        for(String envKey : envKeys){
//          if(envKey.equals("PATH")) continue;
            String value = environmentMap.get(envKey);
            result.append("  ");
            result.append(envKey).append("=").append(value).append("\n");
        }

        result.append("\n");
/*        
        result.append("execパス:\n");
        for(String path : paths){
            result.append("  ");
            result.append(path).append("\n");
        }

        result.append("\n");
*/        
        result.append("クラスパス:\n");
        for(String path : classpaths){
            result.append("  ");
            result.append(path).append("\n");
        }
        
        return result.toString();
    }
    
    /**
     * オプション文字列を解析する。
     * @param args main()に渡されるオプション文字列
     */
    public static void parseOptions(String[] args){
        invokeArgs.clear();
        for(String arg : args){
            if(arg == null) continue;
            if(arg.length() <= 0) continue;
            invokeArgs.add(arg);
        }
        ListIterator<String> iterator = invokeArgs.listIterator();
        
        while(iterator.hasNext()){
            String arg = iterator.next();
            if(   arg.equals("-h")
               || arg.equals("-help")
               || arg.equals("--help")
               || arg.equals("-?")  ){
                needHelp = true;
                needVersion = false;
                continue;
            }
            if( arg.equals("-version") ){
                needVersion = true;
                needHelp = false;
                continue;
            }
            if( arg.equals("-boldMetal") ){
                boldMetal = true;
                continue;
            }
            if( arg.equals("-nosplash") ){
                noSplash = true;
                continue;
            }
            if( arg.equals("-geometry") ){
                if( ! iterator.hasNext() ){
                    optionError(
                            "起動オプション["
                            +arg
                            +"]に引数がありません。");
                    return;
                }
                String geometry = iterator.next();
                Matcher matcher = geomPattern.matcher(geometry);
                if( ! matcher.matches() ){
                    optionError(
                            "起動オプション["
                            +arg
                            +"]の引数形式["
                            +geometry
                            +"]が不正です。" );
                    return;
                }
                String width = matcher.group(1);
                String height = matcher.group(2);
                String xSign = matcher.group(3);
                String xPos = matcher.group(4);
                String ySign = matcher.group(5);
                String yPos = matcher.group(6);
                try{
                    frameWidth = Integer.parseInt(width);
                    frameHeight = Integer.parseInt(height);
                    if(xPos != null && xPos.length() > 0){
                        frameXpos = Integer.parseInt(xPos);
                        if(xSign.equals("-")) frameXpos = -frameXpos;
                    }
                    if(yPos != null && yPos.length() > 0){
                        frameYpos = Integer.parseInt(yPos);
                        if(ySign.equals("-")) frameYpos = -frameYpos;
                    }
                }catch(NumberFormatException e){
                    optionError(
                            "起動オプション["
                            +arg
                            +"]の引数形式["
                            +geometry
                            +"]が不正です。" );
                    return;
                }
                
                continue;
            }
            if( arg.equals("-vminfo") ){
                needVMInfo = true;
                continue;
            }
            if( arg.equals("-consolelog") ){
                consolelog = true;
                continue;
            }
            
            optionError(
                    "未定義の起動オプション["
                    +arg
                    +"]が指定されました。");
            assert false;
        }
        
        return;
    }
    
    /**
     * 起動オプションエラー共通処理
     * @param message エラーメッセージ
     */
    private static void optionError(CharSequence message){
        System.err.println(message);
        System.err.println(
                "起動オプション一覧は、"
                +"起動オプションに「-help」を指定すると確認できます。" );
        Jindolf.runtime.exit(1);
        assert false;
        return;
    }
}
