/*
 * Copyright 2007-2009 Jiemamy Project and the Others.
 * Created on 2008/08/03
 *
 * This file is part of Jiemamy.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.jiemamy.eclipse.action;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.draw2d.geometry.Insets;
import org.eclipse.draw2d.graph.DirectedGraph;
import org.eclipse.draw2d.graph.DirectedGraphLayout;
import org.eclipse.draw2d.graph.Edge;
import org.eclipse.draw2d.graph.EdgeList;
import org.eclipse.draw2d.graph.Node;
import org.eclipse.draw2d.graph.NodeList;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;

import org.jiemamy.Migration;
import org.jiemamy.eclipse.editor.editpart.diagram.AbstractJmNodeEditPart;
import org.jiemamy.eclipse.editor.editpart.diagram.RootEditPart;
import org.jiemamy.model.ConnectionProfile;
import org.jiemamy.model.DiagramPresentationModel;
import org.jiemamy.model.DiagramPresentations;
import org.jiemamy.model.NodeProfile;
import org.jiemamy.model.RootModel;
import org.jiemamy.model.connection.ConnectionAdapter;
import org.jiemamy.model.geometory.JmPoint;
import org.jiemamy.model.geometory.JmRectangle;
import org.jiemamy.model.node.NodeAdapter;
import org.jiemamy.utils.model.PresentationUtil;

/**
 * 自動レイアウトアクション。
 * @author daisuke
 */
public class AutoLayoutAction extends AbstractJiemamyAction {
	
	private static final int PADDING = 40;
	

	private static Node getNode(List<Node> list, NodeAdapter model) {
		for (Node obj : list) {
			EntityNode node = (EntityNode) obj;
			if (node.model == model) {
				return node;
			}
		}
		return null;
	}
	
	/**
	 * インスタンスを生成する。
	 * 
	 * @param viewer ビューア
	 */
	public AutoLayoutAction(GraphicalViewer viewer) {
		super(Messages.AutoLayoutAction_name, viewer);
	}
	
	@Override
	public void run() {
		RootEditPart rootEditPart = (RootEditPart) getViewer().getContents();
		RootModel rootModel = rootEditPart.getModel();
		CompoundCommand commands = new CompoundCommand();
		
		@SuppressWarnings("unchecked")
		// TODO キャスト安全性の根拠提示
		List<EditPart> editParts = rootEditPart.getChildren();
		
		@SuppressWarnings("unchecked")
		// TODO キャスト安全性の根拠提示
		List<Node> graphNodes = new NodeList();
		
		@SuppressWarnings("unchecked")
		// TODO キャスト安全性の根拠提示
		List<Edge> graphEdges = new EdgeList();
		
		// assemble nodes
		for (EditPart obj : editParts) {
			if (obj instanceof AbstractJmNodeEditPart) {
				AbstractJmNodeEditPart editPart = (AbstractJmNodeEditPart) obj;
				NodeAdapter model = editPart.getModel();
				EntityNode node = new EntityNode();
				node.model = model;
				node.width = editPart.getFigure().getSize().width;
				node.height = editPart.getFigure().getSize().height;
				graphNodes.add(node);
			}
		}
		// assemble edges
		for (Object obj : graphNodes) {
			EntityNode node = (EntityNode) obj;
			Collection<ConnectionAdapter> conns = node.model.getSourceConnections();
			CONN_LOOP: for (ConnectionAdapter conn : conns) {
				if (conn.isSelfConnection()) {
					continue;
				}
				
				// skip if the connection already added
				for (Object obj2 : graphEdges) {
					ConnectionEdge edge = (ConnectionEdge) obj2;
					if (edge.model == conn) {
						continue CONN_LOOP;
					}
				}
				Node source = getNode(graphNodes, conn.getSource());
				Node target = getNode(graphNodes, conn.getTarget());
				if (source != null && target != null) {
					graphEdges.add(new ConnectionEdge(source, target, conn));
				}
			}
		}
		DirectedGraph graph = new DirectedGraph();
		graph.setDefaultPadding(new Insets(PADDING));
		graph.nodes = (NodeList) graphNodes;
		graph.edges = (EdgeList) graphEdges;
		new DirectedGraphLayout().visit(graph);
		for (Object obj : graph.nodes) {
			EntityNode node = (EntityNode) obj;
			commands.add(new LayoutCommand(rootModel, Migration.DIAGRAM_INDEX, node.model, node.x, node.y));
		}
		getViewer().getEditDomain().getCommandStack().execute(commands);
	}
	

	private static class ConnectionEdge extends Edge {
		
		private ConnectionAdapter model;
		

		/**
		 * インスタンスを生成する。
		 * 
		 * @param source 接続元ノード
		 * @param target 接続先ノード
		 * @param model コネクションを表すモデル
		 */
		public ConnectionEdge(Node source, Node target, ConnectionAdapter model) {
			super(source, target);
			this.model = model;
		}
	}
	
	private static class EntityNode extends Node {
		
		private NodeAdapter model;
		
	}
	
	/**
	 * Command to relocate the entity model. This command is executed as a part of
	 * CompoundCommand.
	 */
	private static class LayoutCommand extends Command {
		
		private RootModel rootModel;
		
		private final int diagramIndex;
		
		private NodeAdapter target;
		
		private int x;
		
		private int y;
		
		private int oldX;
		
		private int oldY;
		
		private Map<ConnectionAdapter, List<JmPoint>> oldBendpoints = new HashMap<ConnectionAdapter, List<JmPoint>>();
		

		/**
		 * インスタンスを生成する。
		 * 
		 * @param rootModel ルートモデル
		 * @param diagramIndex ダイアグラムエディタのインデックス（エディタ内のタブインデックス）
		 * @param target 対象ノード
		 * @param x X座標
		 * @param y Y座標
		 */
		public LayoutCommand(RootModel rootModel, int diagramIndex, NodeAdapter target, int x, int y) {
			this.rootModel = rootModel;
			this.diagramIndex = diagramIndex;
			this.target = target;
			this.x = x;
			this.y = y;
			DiagramPresentations diagramPresentations = rootModel.getAdapter(DiagramPresentations.class);
			DiagramPresentationModel presentation = diagramPresentations.get(diagramIndex);
			NodeProfile nodeProfile = presentation.getNodeProfiles().get(target);
			JmRectangle boundary = nodeProfile.getBoundary();
			oldX = boundary.x;
			oldY = boundary.y;
		}
		
		@Override
		public void execute() {
			DiagramPresentations diagramPresentations = rootModel.getAdapter(DiagramPresentations.class);
			DiagramPresentationModel presentation = diagramPresentations.get(diagramIndex);
			PresentationUtil.setBoundary(presentation, target, new JmRectangle(x, y, -1, -1));
			oldBendpoints.clear();
			for (ConnectionAdapter conn : target.getSourceConnections()) {
				ConnectionProfile connectionProfile = presentation.getConnectionProfiles().get(conn);
				List<JmPoint> bendpoints = connectionProfile.getBendpoints();
				oldBendpoints.put(conn, new ArrayList<JmPoint>(bendpoints));
				bendpoints.clear();
			}
		}
		
		@Override
		public void undo() {
			DiagramPresentations diagramPresentations = rootModel.getAdapter(DiagramPresentations.class);
			DiagramPresentationModel presentation = diagramPresentations.get(diagramIndex);
			for (ConnectionAdapter conn : target.getSourceConnections()) {
				ConnectionProfile connectionProfile = presentation.getConnectionProfiles().get(conn);
				List<JmPoint> bendpoints = connectionProfile.getBendpoints();
				bendpoints.clear();
				for (JmPoint bendpoint : oldBendpoints.get(conn)) {
					bendpoints.add(bendpoint);
				}
			}
			PresentationUtil.setBoundary(presentation, target, new JmRectangle(oldX, oldY, -1, -1));
		}
	}
}
