/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.webpage;

// java.util
import java.util.HashMap;
import java.util.Iterator;

// javax.servlet
import javax.servlet.http.*;

import org.apache.log4j.Logger;

/**
 *  Stores a map of sessions with other sites for the current portal session.
 *  The map is keyed by the portal session's SessionID (from the servlet api). 
 *  Binds to the servlet's session so that it can be cleaned up on session end.
 *  This session keeps a map of all live sessions per site local session.
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: SessionMap.java,v 1.2 2003/03/04 00:05:14 sgala Exp $ 
 */

public class SessionMap extends HashMap implements HttpSessionBindingListener
{    

    // the name of the user for this session
    private String sessionUser;
    
    // the servlet session id on this server
    private String sessionID;
    
    // the active status of this session
    private boolean active = true;

    // hit count
    private int hitCount = 0;

    // cache hit count 
    private int cacheCount = 0;

    static Logger log = Logger.getLogger(SessionMap.class);

    /**
     * construct a Session Map
     *
     * @param sessionId the sessionID on this host
     * @param sessionUser the user associated with the new session.
     *
     */
    public SessionMap(String sessionID, String sessionUser)
    {
        this.sessionID = sessionID;
        this.sessionUser = sessionUser;
    }


    /**
     * Gets the user associated with this session.
     *
     * @return the user associated with this sessions.
     */
    public String getUser()
    {
        return sessionUser;
    }

    /**
     * Gets the Session ID associated with this session.
     *
     * @return the session ID associated with this sessions.
     */
    public String getSessionID()
    {
        return sessionID;
    }
      
    /**
     * Gets the Session State, for the servlet session.
     *
     * @return the a string describing the session state.
     */
    public String getSessionState()
    {
        return (active) ? "active" : "inactive";
    }

    /**
     * Gets the managed session count for this portal session.
     *
     * @return the managed session count for this session.
     */
    public int getSessionCount()
    {
        return this.size();
    }

    /**
     * Gets the hitcount for this session.
     *
     * @return the hitcount for this session.
     */
    public int getHitCount()
    {
        return hitCount;
    }

    /**
     * Increments the hitcount for this session.
     *
     */
    public void incHitCount()
    {
        hitCount++;
    }

    /**
     * Gets the cache count for this session.
     *
     * @return the cache count for this session.
     */
    public int getCacheCount()
    {
        return cacheCount;
    }

    /**
     * Increments the hitcount for this session.
     *
     */
    public void incCacheCount()
    {
        cacheCount++;
    }


    /**
     * This method is a session binding event callback.
     * It is called when the session is bound
     *
     */
    public void valueBound(HttpSessionBindingEvent event)
    {
    }

    /**
     * This method is a session unbinding event callback.
     * It is called when the session is unbound.
     * We need to clean up all the site sessions associated with this Session.
     * The session is marked as 'not active' and is no longer useable.
     * It session object is then removed from the agent's cache 
     *
     */    
    public void valueUnbound(HttpSessionBindingEvent event)
    {
        log.info("~~~ SessionMap UNBOUND as " + event.getName() + " from " + event.getSession().getId() );

        // Now logout of all sessions
        Iterator it = values().iterator();
        while (it.hasNext())
        {
            SiteSession hps = (SiteSession)it.next();
            try 
            {
                hps.logout(null);

            } catch(Exception e)
            {
                // continue logging out even if one fails
                log.error("Unbound-Logout of Session: " + e);                                
            }            
        }
        active = false;
        clear();
    }


 }
