/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.threadpool;

//turbine stuff
import org.apache.turbine.services.TurbineServices;

// Jetspeed classes
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;

/**
 * A thread that is used to process Runnables.  This thread will wait until it is 
 * notified by another thread that it needs processing.  However it will only 
 * process if getRunnable != null.
 *
 * @author <a href="mailto:burton@apache.org">Kevin A. Burton</a>
 * @author <a href="mailto:sgala@apache.org">Santiago Gala</a>
 * @version $Id: RunnableThread.java,v 1.6 2003/07/23 19:50:26 morciuch Exp $
 */
public class RunnableThread extends Thread 
{
    /**
     * Static initialization of the logger for this class
     */    
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(RunnableThread.class.getName());
    
    private boolean running = false;
    private static int next_id = 0;

    /**
     * Increment a counter so that we can identify threads
     * easily.
     */
    private static synchronized int getNextId() {
        return ++next_id;
    };
    private int id = 0;

    /**
    The runnable that you want to process
    */
    private Runnable runnable = null;
    
    public RunnableThread()
    {
        super();
        this.setDaemon(true);
    }

    /**
     * Creates a new Thread in the specified threadgroup
     *
     * @param tg the Threadgroup which will contain the new Thread
     */
    public RunnableThread( ThreadGroup tg) {
        super( tg, "Provisory");
        this.id = getNextId();
        super.setName("RunnableThread:" + this.id );
        this.setPriority( Thread.MIN_PRIORITY );
        this.setDaemon(true);

    }

    /**
     * Creates a new Thread in the specified threadgroup and assigns
     * it an id.
     *
     * @param tg the Threadgroup which will contain the new Thread
     * @param id an identifier for the new Thread
     */
    public RunnableThread( ThreadGroup tg, 
                           int id ) {

        super( tg, "RunnableThread:" + id );
        this.setPriority( Thread.MIN_PRIORITY );
        this.setDaemon(true);       
        this.id = id;
    }
    
    /**
     * Processes the Runnable object assigned to it, whenever one
     * is available
     */
    public void run() {

        /*
          FIXME: move to a static class variable to allow for pool shutdown
        */
        boolean poolrunning  = true;
        while ( poolrunning ) {


            //On creation, we are idle.
            //So, add ourselves to the Pool.
            //Next times we come here, we are just finished
            //one run...
            this.setRunning( false );                

            this.setRunnable( null );
                
            synchronized( this ) {
                if( this.getPriority() != 
                    JetspeedThreadPoolService.DEFAULT_THREAD_PRIORITY ) {
                    //give the thread back it's default priority.
                    this.setPriority( JetspeedThreadPoolService.DEFAULT_THREAD_PRIORITY );
                }
                
                //SGP I don't think it is needed. The scheduler will do its job
                //and the thread will be released sooner. Later, it will wait
                //until the Pool reuses it. Correct me if I'm wrong
                //but please comment the reasons, as I don't get it :)

                //yield this thread so that other threads can now execute
                //if necessary.
                //this.yield();

                //ok... add this thread back into the thread pool
                ( (JetspeedThreadPoolService)TurbineServices
                  .getInstance()
                  .getService( ThreadPoolService.SERVICE_NAME ) )
                    .release( this );
                
            
                //if the runnable == null wait because it has been not been
                //directly assigned a task.. 
                if ( this.getRunnable() == null ) {
               
                    try {
                        this.wait();
                    } catch (InterruptedException e) {
                        //this is a normal situation.  
                        //the DaemonFactory may want to stop this thread form 
                        //sleeping and call interrupt() on this thread.
                    } catch ( Throwable t ) {
                        logger.error("Throwable",  t);
                        //continue;
                    }

                }

            }

            
            if ( this.getRunnable() != null ) {
                this.setRunning( true );

                try {
                    
                    this.getRunnable().run();

                } catch ( Throwable t ) {
                    logger.error( "A problem occured while trying to run your thread", t );
                }
                    
            }
            
        }
        
    }
    
    //misc getters/setters

    /**
     * Set the Runnable process to execute
     *
     * @param runnable the Object to execute
     */
    public void setRunnable( Runnable runnable ) {
        this.runnable = runnable;
    }

    /**
     * Get the Runnable process executing
     *
     * @return the Object executed by this thread
     */
    public Runnable getRunnable() {
        return this.runnable;
    }
    
    /**
     * Test whether the thread is currently executing a process
     *
     * @return the status of this thread. If true, the thread is currently 
     * executing a Runnable process, if false it's waiting for a new process
     */
    private boolean isRunning() {
        return this.running;
    }
    
    /**
     * Set the running status of this thread.
     *
     * @return the status of this thread
     */
    private void setRunning( boolean running ) {
        this.running = running;
    }
    
    /**
     * Get the numeric identifier of this thread
     *
     * @return the identifier of the thread
     */
    public int getId() {
        return this.id;
    }
}
