/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.template;

import org.apache.turbine.services.Service;
import org.apache.turbine.util.RunData;

import java.util.Locale;

/**
 * <p>This interface is a facade for all template location related operations.
 * Template location algorithms are different from the Velocity template location,
 * since Jetspeed has a specialized template directory structure.
 * This is a fix to get us through unti the TurbineTemplateService can locate
 * resources by NLS and mediatype. Then it can be removed</p>
 *
 * <p>The directory structure is currently layout out in the following order:
 *    /templateType/mediaType/LanguageCode/CountryCode</p>
 * <p>Example: /screens/html/en/US/resource.vm</p>
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @author <a href="mailto:paulsp@apache.org">Paul Spencer</a>
 * @author <a href="mailto:kimptoc_mail@yahoo.com">Chris Kimpton</a>
 * @version $Id: TemplateLocatorService.java,v 1.7 2003/03/04 00:05:13 sgala Exp $
 */

public interface TemplateLocatorService extends Service
{
 
    /** The name of this service */
    public String SERVICE_NAME = "TemplateLocator";

    /**
     * Locate a screen template using Jetspeed template location algorithm, searching by
     * mediatype and language criteria extracted from the request state in rundata.
     *
     * @param data The rundata for the request.
     * @param template The name of the template.
     *
     * @return The path relative to the screens directory for the requested screen template,
     *          or null if not found.
     */
    public String locateScreenTemplate(RunData data, String template);

    /**
     * Locate a layout template using Jetspeed template location algorithm, searching by
     * mediatype and language criteria extracted from the request state in rundata.
     *
     * @param data The rundata for the request.
     * @param template The name of the template.
     *
     * @return The path relative to the layouts directory for the requested layout template,
     *          or null if not found.
     */
    public String locateLayoutTemplate(RunData data, String template);

    /**
     * Locate a portlet template using Jetspeed template location algorithm, searching by
     * mediatype and language criteria extracted from the request state in rundata.
     *
     * @param data The rundata for the request.
     * @param template The name of the template.
     *
     * @return The path relative to the portlets directory for the requested portlet template,
     *          or null if not found.
     */
    public String locatePortletTemplate(RunData data, String template);

    /**
     * Locate a control template using Jetspeed template location algorithm, searching by
     * mediatype and language criteria extracted from the request state in rundata.
     *
     * @param data The rundata for the request.
     * @param template The name of the template.
     *
     * @return The path relative to the controls directory for the requested control template,
     *          or null if not found.
     */
    public String locateControlTemplate(RunData data, String template);

    /**
     * Locate a controller template using Jetspeed template location algorithm, searching by
     * mediatype and language criteria extracted from the request state in rundata.
     *
     * @param data The rundata for the request.
     * @param template The name of the template.
     *
     * @return The path relative to the controllers directory for the requested controller template,
     *          or null if not found.
     */
    public String locateControllerTemplate(RunData data, String template);

    /**
     * Locate a navigation template using Jetspeed template location algorithm, searching by
     * mediatype and language criteria extracted from the request state in rundata.
     *
     * @param data The rundata for the request.
     * @param template The name of the template.
     *
     * @return The path relative to the controllers directory for the requested controller template,
     *          or null if not found.
     */
    public String locateNavigationTemplate(RunData data, String template);

    /**
     * Locate an email template using Jetspeed template location algorithm, searching by
     * language criteria extracted from the request state in rundata.
     *
     * @param data The rundata for the request.
     * @param template The name of the template.
     *
     * @return The path relative to the emails directory for the requested email template,
     *          or null if not found.
     */
    public String locateEmailTemplate(RunData data, String template);

    /**
     * Locate an email template using Jetspeed template location algorithm
     *
     * @param data The rundata for the request.
     * @param template The name of the template.
     * @param locale The locale of the template.
     *
     * @return The path relative to the emails directory for the requested email template,
     *          or null if not found.
     */
    public String locateEmailTemplate(RunData data, String template, Locale locale);

    /**
     * Locate a parameter style template using Jetspeed template location algorithm, searching by
     * mediatype and language criteria extracted from the request state in rundata.
     *
     * @param data The rundata for the request.
     * @param template The name of the template.
     *
     * @return The path relative to the parameters directory for the requested portlet template,
     *          or null if not found.
     */
    public String locateParameterTemplate(RunData data, String template);

}
