/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security;

import java.util.Iterator;
import java.util.HashMap;

// Junit imports
import junit.framework.Test;
import junit.framework.TestSuite;

import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

// Jetspeed imports
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.jetspeed.om.security.Group;
import org.apache.jetspeed.om.security.JetspeedGroupFactory;

/**
 * Unit test for GroupManagement interface
 * 
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: TestGroupManagement.java,v 1.5 2003/08/08 18:28:03 morciuch Exp $
 */

public class TestGroupManagement extends JetspeedTestCase {    
    
    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestGroupManagement( String name ) {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[]) 
    {
        junit.awtui.TestRunner.main( new String[] { TestGroupManagement.class.getName() } );
    }
 
    public void setup() 
    {
        //System.out.println("Setup: Testing Turbine Group Management");         
    }

    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite() 
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestGroupManagement.class );
    }

    /**
     * Tests getGroups method
     * @throws Exception
     */

    public void testGetGroups() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;
        HashMap map = new HashMap();

        try
        {
            Iterator groups = service.getGroups();
            while (groups.hasNext())
            {
                group = (Group)groups.next();
                map.put(group.getName(), group);
            }
            assertTrue(map.get("apache") != null);
            assertTrue(map.get("Jetspeed") != null);
            assertTrue(map.get("bogusGroup") == null);
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getGroups Test OK ");

    }

    /**
     * Tests getGroups method
     * @throws Exception
     */

    public void testGetGroupsForUser() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;
        HashMap map = new HashMap();

        try
        {
            Iterator groups = service.getGroups("turbine");
            while (groups.hasNext())
            {
                group = (Group)groups.next();
                map.put(group.getName(), group);
                System.out.println("[turbine] group = " + group.getName());
            }
            assertTrue(map.get("Jetspeed") != null);
            assertTrue(map.get("apache") == null);

            map.clear();
            groups = service.getGroups("admin");            
            while (groups.hasNext())
            {
                group = (Group)groups.next();
                map.put(group.getName(), group);
                System.out.println("[admin] group = " + group.getName());
            }
            assertTrue(map.get("Jetspeed") != null);

        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getGroups Test OK ");

    }

    /**
     * Tests addGroup method 
     * @throws Exception
     */

    public void testAddGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            group = JetspeedGroupFactory.getInstance();
            group.setName("bogus");
            service.addGroup(group);
            System.out.println("new group id = " + group.getId());
            assertTrue(group.getId() != null);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            group = JetspeedGroupFactory.getInstance();
            group.setName("bogus");
            service.addGroup(group);
            fail("Should've thrown a dup key exception on group");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof GroupException);           
        }

        System.out.println("Completed addGroup Test OK ");

    }

    /**
     * Tests getRemoveGroup method 
     * @throws Exception
     */

    public void testRemoveGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            service.removeGroup("bogus");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.removeGroup("catchmeifyoucan");
            fail("Should've thrown a not found exception on group");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof GroupException);
        }

        System.out.println("Completed addGroup Test OK ");

    }

    /**
     * Tests getGroup method
     * @throws Exception
     */

    public void testGetGroup() throws Exception 
    {
        GroupManagement service = getService();

        try
        {
            Group group = service.getGroup("Jetspeed");
            System.out.println("*** group nm = " + group.getName());
            System.out.println("*** group id = " + group.getId());
            assertTrue(group.getName().equals("Jetspeed"));
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getGroup Test OK ");

    }

    /**
     * Tests saveGroup method 
     * @throws Exception
     */

    public void testSaveGroup() throws Exception 
    {
        GroupManagement service = getService();

        try
        {
            Group group = service.getGroup("apache");
            service.saveGroup(group);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed saveGroup Test OK ");

    }

    /**
     * Tests joinGroup method 
     * @throws Exception
     */
    public void testJoinGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            service.joinGroup("turbine", "apache");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.joinGroup("baduser", "apache");
            fail("Should've thrown a bad user exception on join");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof GroupException);           
        }

        System.out.println("Completed joinGroup Test OK ");

    }

    /**
     * Tests unjoinGroup method 
     * @throws Exception
     */
    public void testUnjoinGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            service.unjoinGroup("turbine", "apache");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.unjoinGroup("baduser", "apache");
            fail("Should've thrown a bad user exception on unjoin");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof GroupException);           
        }

        System.out.println("Completed unjoinGroup Test OK ");

    }

    /**
     * Tests inGroup method 
     * @throws Exception
     */
    public void testInGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            boolean in = service.inGroup("admin", "Jetspeed");
            assertTrue(true == in);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            boolean in = service.inGroup("turbine", "apache");
            assertTrue(false == in);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed inGroup Test OK ");

    }

  /*
    Configuration object to run Turbine outside a servlet container
    ( uses turbine.properties )
    */
    private static TurbineConfig config = null;
    
    /**
    Sets up TurbineConfig using the system property:
    <pre>turbine.properties</pre>
    */
    static
    {
        try
        {
            config = new TurbineConfig( "../webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }

    private static GroupManagement getService()
    {
        return (GroupManagement)TurbineServices
                .getInstance()
                .getService(GroupManagement.SERVICE_NAME);
    }

}






