/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security;

import org.apache.jetspeed.services.rundata.JetspeedRunData;
import org.apache.turbine.services.Service;

import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.portal.Portlet;

/**
 * The Security Service manages Users, Groups Roles and Permissions in the 
 * system. The Jetspeed Security Service extends the interface of the Turbine
 * Security Service, adding on the Jetspeed specific interface: AccessControl
 * for controlling access to portal resources (portlets, panes).
 *
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: JetspeedSecurityService.java,v 1.10 2003/03/04 00:05:10 sgala Exp $
 */


public interface JetspeedSecurityService extends Service
{
   /** The name of this service */
   public String SERVICE_NAME = "JetspeedSecurity";

   //////////////////////////////////////////////////////////////////////////
   //
   // Required JetspeedSecurity Functions
   //
   // Required Features provided by default JetspeedSecurity
   //
   //////////////////////////////////////////////////////////////////////////

   /*
    * Factory to create a new JetspeedUser, using JetspeedUserFactory.
    * The class that is created by the default JetspeedUserFactory is configured
    * in the JetspeedSecurity properties:
    *
    *    services.JetspeedSecurity.user.class=
    *        org.apache.jetspeed.om.security.BaseJetspeedUser
    *
    * @return JetspeedUser a newly created user that implements JetspeedUser.
    */
   public JetspeedUser getUserInstance();


    //////////////////////////////////////////////////////////////////////////
    //
    // Optional JetspeedSecurity Features 
    //
    // Features are not required to be implemented by Security Provider
    //
    //////////////////////////////////////////////////////////////////////////

    /*
     * During logon, the username can be case sensitive or case insensitive.
     *
     * Given a username, converts the username to either lower or upper case.
     * This optional feature is configurable from the JetspeedSecurity.properties:
     *
     *     <code>services.JetspeedSecurity.caseinsensitive.username = true/false</code>
     *     <code>services.JetspeedSecurity.caseinsensitive.upper = true/false</code>
     *
     * If <code>caseinsensitive.username</code> is true,  
     * then conversion is enabled and the username will be converted before 
     * being sent to the Authentication provider.
     *
     * @param username The username to be converted depending on configuration.
     * @return The converted username.
     *
     */
    public String convertUserName(String username);

    /*
     * During logon, the password can be case sensitive or case insensitive.
     *
     * Given a password, converts the password to either lower or upper case.
     * This optional feature is configurable from the JetspeedSecurity.properties:
     *
     *     <code>services.JetspeedSecurity.caseinsensitive.password = true/false</code>
     *     <code>services.JetspeedSecurity.caseinsensitive.upper = true/false</code>
     *
     * If <code>caseinsensitive.password</code> is true,  
     * then conversion is enabled and the password will be converted before 
     * being sent to the Authentication provider.
     *
     * @param password The password to be converted depending on configuration.
     * @return The converted password.
     *
     */
    public String convertPassword(String password);

    /*
     * Logon Failure / Account Disabling Feature
     *
     * Checks and tracks failed user-logon attempts.
     * If the user fails to logon after a configurable number of logon attempts,
     * then the user's account will be disabled.
     *
     * This optional feature is configurable from the JetspeedSecurity.properties:
     *
     *     <code>services.JetspeedSecurity.logon.auto.disable=false</code>
     *
     * The example setting below allows for 3 logon strikes per 300 seconds.
     * When the strike.count is exceeded over the strike.interval, the account
     * is disabled. The strike.max is the cumulative maximum.
     *
     *     <code>services.JetspeedSecurity.logon.strike.count=3</code>
     *     <code>services.JetspeedSecurity.logon.strike.interval=300</code>
     *     <code>services.JetspeedSecurity.logon.strike.max=10</code>
     *
     * These settings are not persisted, and in a distributed environment are 
     * only tracked per node.
     *
     * @param username The username to be checked.
     * @return True if the strike count reached the maximum threshold and the
     *         user's account was disabled, otherwise False.
     *
     */
    public boolean checkDisableAccount(String username);

    /*
     * Logon Failure / Account Disabling Feature
     *    
     * Returns state of the the logon failure / account disabling feature.
     * 
     * If the user fails to logon after a configurable number of logon attempts,
     * then the user's account will be disabled.
     *
     * @see JetspeedSecurityService#checkLogonFailures
     *
     * @return True if the feature is enabled, false if the feature is disabled.
     *
     */
    public boolean isDisableAccountCheckEnabled();

    /*
     * Logon Failure / Account Disabling Feature
     *    
     * Resets counters for the logon failure / account disabling feature.
     * 
     * If the user fails to logon after a configurable number of logon attempts,
     * then the user's account will be disabled.
     *
     * @see JetspeedSecurityService#checkLogonFailures
     *
     * @param username The username to reset the logon failure counters.
     *
     */
    public void resetDisableAccountCheck(String username);


    //////////////////////////////////////////////////////////////////////////
    //
    // Optional JetspeedSecurity Helpers
    //
    //////////////////////////////////////////////////////////////////////////


    /**
     * Helper to UserManagement.
     * Retrieves a <code>JetspeedUser</code> given the primary principle username.
     * The principal can be any valid Jetspeed Security Principal:
     *   <code>org.apache.jetspeed.om.security.UserNamePrincipal</code>
     *   <code>org.apache.jetspeed.om.security.UserIdPrincipal</code>
     *   
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param username The username principal.
     * @return a <code>JetspeedUser</code> associated to the principal identity.
     * @exception UserException when the security provider has a general failure retrieving a user.
     * @exception UnknownUserException when the security provider cannot match
     *            the principal identity to a user.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */

    public JetspeedUser getUser(String username) 
        throws JetspeedSecurityException;


    /**
     * Helper to PortalAuthorization.
     * Gets a <code>JetspeedUser</code> from rundata, authorize user to perform the secured action on
     * the given <code>Portlet</code> resource. If the user does not have
     * sufficient privilege to perform the action on the resource, the check returns false,
     * otherwise when sufficient privilege is present, checkPermission returns true.
     *
     * @param rundata request that the user is taken from rundatas
     * @param action the secured action to be performed on the resource by the user.     
     * @param portlet the portlet resource.
     * @return boolean true if the user has sufficient privilege.
     */
    public boolean checkPermission(JetspeedRunData runData, String action, Portlet portlet);

    /**
     * Helper to PortalAuthorization.
     * Gets a <code>JetspeedUser</code> from rundata, authorize user to perform the secured action on
     * the given <code>Entry</code> resource. If the user does not have
     * sufficient privilege to perform the action on the resource, the check returns false,
     * otherwise when sufficient privilege is present, checkPermission returns true.
     *
     * @param rundata request that the user is taken from rundatas
     * @param action the secured action to be performed on the resource by the user.     
     * @param entry the portal entry resource.
     * @return boolean true if the user has sufficient privilege.
     */
    //public boolean checkPermission(JetspeedRunData runData, String action, RegistryEntry entry);

   /*
     * Security configuration setting to disable all action buttons for the Anon user
     * This setting is readonly and is edited in the JetspeedSecurity deployment
     *    
     *
     * @return True if the feature actions are disabled for the anon user
     *
     */
    public boolean areActionsDisabledForAnon();

    /*
     * Security configuration setting to disable all action buttons for all users
     * This setting is readonly and is edited in the JetspeedSecurity deployment
     *    
     *
     * @return True if the feature actions are disabled for the all users
     *
     */
    public boolean areActionsDisabledForAllUsers();


   /*
     * Gets the name of the anonymous user account if applicable
     *    
     *
     * @return String the name of the anonymous user account
     *
     */
    public String getAnonymousUserName();


}

