/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.registry;

// Java classes
import java.util.Hashtable;
import java.util.Enumeration;
import javax.servlet.ServletConfig;

//turbine stuff
import org.apache.turbine.services.InitializationException;
import org.apache.turbine.services.TurbineBaseService;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.services.resources.ResourceService;

// Jetspeed classes
import org.apache.jetspeed.om.registry.Registry;
import org.apache.jetspeed.om.registry.RegistryEntry;
import org.apache.jetspeed.om.registry.RegistryException;
import org.apache.jetspeed.om.registry.base.BaseRegistry;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;

/**
 * <p>This is an implementation of the <code>RegistryService</code>
 * based on the Jetspeed Database Persistence Manager</p>
 *
 * <p>This service expects the following properties to be set for correct operation:
 * <dl>
 *    <dt>none</dt><dd></dd>
 *
 * </dl>
 * </p>
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: DatabaseRegistryService.java,v 1.4 2003/07/23 19:50:22 morciuch Exp $
 */
public class DatabaseRegistryService
        extends TurbineBaseService
        implements RegistryService
{
    /**
     * Static initialization of the logger for this class
     */    
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(DatabaseRegistryService.class.getName());
    
    /** regsitry type keyed list of entries */
    private Hashtable registries = new Hashtable();


    /**
     * Returns a Registry object for further manipulation
     *
     * @param regName the name of the registry to fetch
     * @return a Registry object if found by the manager or null
     */
    public Registry get( String regName )
    {
        return (Registry)registries.get( regName );
    }

    /**
     *  List all the registry currently available to this service
     *
     * @return an Enumeration of registry names.
     */
    public Enumeration getNames()
    {
        return registries.keys();
    }

    /**
     * Creates a new RegistryEntry instance compatible with the current
     * Registry instance implementation
     *
     * @param regName the name of the registry to use
     * @return the newly created RegistryEntry
     */
    public RegistryEntry createEntry(String regName)
    {
		RegistryEntry entry = null;
        Registry registry = (Registry) registries.get(regName);

        if (registry != null)
        {
			entry = registry.createEntry();
        }

		return entry;
	}

    /**
     * Returns a RegistryEntry from the named Registry.
     * This is a convenience wrapper around {@link
     * org.apache.jetspeed.om.registry.Registry#getEntry }
     *
     * @param regName the name of the registry
     * @param entryName the name of the entry to retrieve from the
     *                  registry
     * @return a RegistryEntry object if the key is found or null
     */
    public RegistryEntry getEntry( String regName, String entryName )
    {
        try
        {
            return ((Registry)registries.get( regName )).getEntry( entryName );
        }
        catch ( RegistryException e )
        {
            logger.info("Failed to retrieve "+entryName+" from "+regName);
        }
        catch ( NullPointerException e )
        {
            logger.error(regName + " registry is not known ", e);
        }

        return null;
    }

    /**
     * Add a new RegistryEntry in the named Registry.
     * This is a convenience wrapper around {@link
     * org.apache.jetspeed.om.registry.Registry#addEntry }
     *
     * @param regName the name of the registry
     * @param entry the Registry entry to add
     * @exception Sends a RegistryException if the manager can't add
     *            the provided entry
     */
    public void addEntry(String regName, RegistryEntry entry) throws RegistryException
    {
        if (entry == null)
        {
            return;
        }

        BaseRegistry registry = (BaseRegistry)registries.get( regName );

        if (registry!=null)
        {
            if (this.getEntry(regName, entry.getName())!=null)
            {
                registry.setLocalEntry(entry);
            }
            else
            {
                registry.addLocalEntry(entry);
            }
        }
    }

    /**
     * Deletes a RegistryEntry from the named Registry
     * This is a convenience wrapper around {@link
     * org.apache.jetspeed.om.registry.Registry#removeEntry }
     *
     * @param regName the name of the registry
     * @param entryName the name of the entry to remove
     */
    public void removeEntry( String regName, String entryName )
    {
        if (entryName==null)
        {
            return;
        }

        BaseRegistry registry = (BaseRegistry)registries.get( regName );

        if (registry!=null)
        {
            // the entry is physically removed, remove the dangling reference
            registry.removeLocalEntry( entryName );
        }
    }

    /**
     * This is the early initialization method called by the
     * Turbine <code>Service</code> framework
     */
    public void init( ServletConfig conf ) throws InitializationException
    {

        ResourceService serviceConf = ((TurbineServices)TurbineServices.getInstance())
                                                     .getResources(RegistryService.SERVICE_NAME);

        // read the configuration keys
        try
        {

        }
        catch ( Throwable t )
        {
            throw new InitializationException( "Unable to initialize DatabaseRegistryService, missing config keys");
        }

        //Mark that we are done
        setInit(true);

        // load the registries
        // TODO: write this
    }


    /** Late init method from Turbine Service model */
    public void init( ) throws InitializationException
    {
        logger.debug( "Late init for Registry called" );
        while( !getInit() )
        {
            //Not yet...
            try
            {
                Thread.sleep( 500 );
                logger.debug( "Waiting for init of Registry..." );
            }
            catch (InterruptedException ie )
            {
                logger.error("Exception",  ie);
            }
        }
        logger.debug( "We are done! (Registry)" );
    }


    /**
     * This is the shutdown method called by the
     * Turbine <code>Service</code> framework
     */
    public void shutdown()
    {
    }

}
