/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portal;

import java.util.Iterator;

import org.apache.jetspeed.om.profile.Entry;
import org.apache.jetspeed.om.profile.PSMLDocument;
import org.apache.jetspeed.om.profile.Profile;
import org.apache.jetspeed.portal.Portlet;

/**
 * This interface provides an easy, object-oriented approach to modifing 
 * a portlet instance's persistent attributes.
 * It provides methods for getting, setting and removing attributes from 
 * portlet instance persistence storage. 
 *
 * In a Jetspeed 1.4x PSML profile, the default XML format for an instance and attribute is:
 *
 * <entry>
 *    <parameter name="someName" value="someValue"/>
 *
 * @author <a href="mailto:sweaver@rippe.com">Scott Weaver</a>
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a> 
 * @version $Id: PortletInstance.java,v 1.2 2003/03/04 00:05:03 sgala Exp $
 */
public interface PortletInstance
{
    /**
     * Retrieves an attributes value associated with a named object from a
     * portlet instance's persistence storage. 
     *
     * @param name The name of the attribute
     * @param defaultValue The default value if the attribute is not found.
     * @return String The attribute value, or the defaultValue if not found.
     */
    String getAttribute(String name, String defaultValue);
    
    /**
     * Retrieves an attributes value associated with a named object from a
     * portlet instance's persistence storage. 
     *
     * @param name The name of the attribute
     * @return String The attribute value, or the empty string if not found.
     */
    String getAttribute(String name);


    /**
     * Sets a portlet instance attribute into persistence storage.
     *
     * @param name The name of the attribute.
     * @param value The value of the attribute.
     */
    void setAttribute(String name, String value);
    
    /**
     * Removes a portlet instance attribute from persistence storage.
     *
     * @param name The name of the attribute.
     */
    void removeAttribute(String name);
    
    /**
     * Removes all portlet instance attributes from persistence storage.
     *
     */
    void removeAllAttributes();
    
    /**
     * Retrieves a list of all of the attributes of this portlet instance
     *  as <code>org.apache.jetspeed.om.profile.Parameter</code> objects.
     *
     * @return java.util.Iterator
     */
    Iterator getAttributes();
    
    /**
     * Retrieves a list of all attributes names for all the attributes
     * contained within this portlet instance.
     *
     * @return java.util.Iterator
     */
    Iterator getAttributeNames();
    
    /**
     * Returns the PSMLDocument that contains this portlet instance.
     *
     * @return org.apache.jetspeed.om.profile.PSMLDocument
     * 
     */
    PSMLDocument getDocument();
    
    /**
     * Returns the Profile instance containing this portlet instance.  
     *
     * @return org.apache.jetspeed.om.profile.Profile
     */
    Profile getProfile();
    
    /**
     * Returns the PSML OM instance associated with this instance.
     *
     * @return org.apache.jetspeed.om.profile.Entry
     */
    Entry getEntry();    
    
     /**
      * Returns the parent portlet of this PortletInstance.
      *
      * @return org.apache.jetspeed.portal.Portlet 
      */
    Portlet getPortlet();    
    
    /**
     * Retrieves the name of the parent portlet as it is defined within the portlet registry.
     *
     * @return String The name of the portlet.
     */
    String getName();
    
    /**
     * Retrieves the title (if it is defined) of the parent portlet as it is defined 
     * within the portlet registry.
     *
     * @return String The title of the portlet.
     */
    String getTitle();
    
    /**
     * Retrieves the description (if it is defined) of the parent portlet as it 
     * is defined within the portlet registry.
     * 
     * @return String The description of the portlet.
     */
    String getDescription();   
    

}
