/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.modules.actions.portlets;

// Turbine stuff
import org.apache.turbine.util.RunData;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.Comparable;
import org.apache.turbine.util.QuickSort;

// Jetspeed stuff
import org.apache.jetspeed.portal.Portlet;
import org.apache.jetspeed.modules.actions.portlets.JspPortletAction;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.webservices.finance.stockmarket.StockQuoteService;
import org.apache.jetspeed.webservices.finance.stockmarket.StockQuote;
import org.apache.jetspeed.util.PortletConfigState;
import org.apache.jetspeed.util.PortletSessionState;
import org.apache.jetspeed.util.StringUtils;

/**
 * This action sets up the template context for retrieving stock quotes.
 *
 * @author <a href="mailto:morciuch@apache.org">Mark Orciuch</a>
 * @version $Id: JspStockQuoteAction.java,v 1.2 2003/07/23 19:50:12 morciuch Exp $ 
 */

public class JspStockQuoteAction extends JspPortletAction implements Comparable
{
    private static final String SYMBOLS = "symbols";
    private static final String COLUMNS = "columns";
    private static final String QUOTES = "quotes";
    private static final String SORT = "sort";
    private static final String SELECTED_COLUMNS = "selected-columns";
    private static final String[] ALL_COLUMNS = {"Symbol","Price","Change","Volume"};
    private String sort = null;

    /**
     * Static initialization of the logger for this class
     */    
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(JspStockQuoteAction.class.getName());     
    
    /**
     * Build the normal state content for this portlet.
     *
     * @param portlet The jsp-based portlet that is being built.
     * @param rundata The turbine rundata context for this request.
     */
    protected void buildNormalContext(Portlet portlet, RunData rundata)
    {

        // We always fetch the most current quotes so might as well call refresh from here
        this.doRefresh(rundata, portlet);
    }

    /**
     * Sort the quotes.
     *
     * @param portlet The jsp-based portlet that is being built.
     * @param rundata The turbine rundata context for this request.
     */
    public void doSort(RunData rundata, Portlet portlet)
    {
        // We always fetch the most current quotes so might as well call refresh from here
        this.doRefresh(rundata, portlet);
        logger.info("JspStockQuoteAction: sorting...");
    }

    /**
     * Refresh the portlet content.
     *
     * @param portlet The jsp-based portlet that is being built.
     * @param rundata The turbine rundata context for this request.
     */
    public void doRefresh(RunData rundata, Portlet portlet)
    {
        try
        {
            // Get reference to stock quote web service
            StockQuoteService service = (StockQuoteService) TurbineServices.getInstance().
                getService(StockQuoteService.SERVICE_NAME);

            // Retrieve portlet parameters
            String symbols = (String) PortletSessionState.getAttributeWithFallback(portlet, rundata, SYMBOLS);

            this.sort = (String) PortletSessionState.getAttributeWithFallback(portlet, rundata, SORT);
            if (this.sort != null)
            {
                PortletSessionState.setAttribute(portlet, rundata, SORT, sort);
                rundata.getRequest().setAttribute(SORT, sort);
            }

            String columns = PortletConfigState.getParameter(portlet, rundata, COLUMNS, 
                                                             StringUtils.arrayToString(ALL_COLUMNS, ","));
            String[] selectedColumnsArray = StringUtils.stringToArray(columns, ",");


            // Request stock quote(s) from the stock quote web service
            String[] symbolArray = StringUtils.stringToArray(symbols, ",");
            StockQuote[] quotes = service.fullQuotes(symbolArray);

            // Sort the entries
            if (this.sort != null)
            {
                QuickSort.quickSort(quotes, 0, quotes.length - 1, this);
                rundata.getRequest().setAttribute(SORT, this.sort);
            }

            // Place appropriate objects in jsp context
            rundata.getRequest().setAttribute(QUOTES, quotes);
            rundata.getRequest().setAttribute(COLUMNS, selectedColumnsArray);
            rundata.getRequest().setAttribute(SELECTED_COLUMNS, columns);

            logger.info("JspStockQuoteAction: refreshing...");
        }
        catch (Exception e)
        {
            logger.error("Exception", e);
        }
    }

    /**
     * Compare to another <code>StockQuote</code>.  Used by the
     * <code>QuickSort</code> class to determine sort order.
     * 
     * @param entry1 The first <code>StockQuoteEntry</code> object.
     * @param entry2 The second <code>StockQuoteEntry</code> object.
     * @return An <code>int</code> indicating the result of the comparison.
     */
    public int compare(Object entry1, Object entry2)
    {
        if (this.sort.equalsIgnoreCase("price"))
        {
            Float entrycol1 = new Float(((StockQuote) entry1).getPrice());
            Float entrycol2 = new Float(((StockQuote) entry2).getPrice());
            return entrycol1.compareTo(entrycol2);
        }
        else if (this.sort.equalsIgnoreCase("symbol"))
        {
            String entrycol1 = ((StockQuote) entry1).getSymbol();
            String entrycol2 = ((StockQuote) entry2).getSymbol();
            return entrycol1.compareTo(entrycol2);
        }
        else if (this.sort.equalsIgnoreCase("change"))
        {
            Double entrycol1 = new Double(((StockQuote) entry1).getChange());
            Double entrycol2 = new Double(((StockQuote) entry2).getChange());
            return entrycol1.compareTo(entrycol2);
        }
        else
        {
            Long entrycol1 = new Long(((StockQuote) entry1).getVolume());
            Long entrycol2 = new Long(((StockQuote) entry2).getVolume());
            return entrycol1.compareTo(entrycol2);
        }

    }

}

