/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.jetspeed.modules.actions.portlets;

// Jetspeed classes
import org.apache.jetspeed.modules.actions.portlets.security.SecurityConstants;
import org.apache.jetspeed.om.registry.ClientEntry;
import org.apache.jetspeed.om.registry.RegistryEntry;
import org.apache.jetspeed.portal.portlets.VelocityPortlet;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.Registry;

// Regexp classes
import org.apache.regexp.RE;
import org.apache.regexp.RESyntaxException;

// Trubine classes
import org.apache.turbine.util.DynamicURI;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

/**
 * This action enables to update the client entries
 *
 * @author <a href="mailto:caius1440@hotmail.com">Jeremy Ford</a>
 * @version $Id: ClientUpdateAction.java,v 1.3 2003/07/23 19:50:12 morciuch Exp $
 */
public class ClientUpdateAction extends RegistryUpdateAction
{
    private static final String whoAmI = "ClientUpdateAction: ";
    private static final String CLIENT_UPDATE_PANE = "ClientForm";
    
    /**
     * Static initialization of the logger for this class
     */    
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(ClientUpdateAction.class.getName());
    
    public ClientUpdateAction()
    {
        registryEntryName = "client_name";
        registry = Registry.CLIENT;
        pane = CLIENT_UPDATE_PANE;
    }

	/**
	 * Subclasses must override this method to provide default behavior
	 * for the portlet action
	 */
	/**
	 * Build the normal state content for this portlet.
	 *
	 * @param portlet The velocity-based portlet that is being built.
	 * @param context The velocity context for this request.
	 * @param rundata The turbine rundata context for this request.
	 */
	protected void buildNormalContext(
		VelocityPortlet portlet,
		Context context,
		RunData rundata)
		throws Exception
	{
        super.buildNormalContext(portlet, context, rundata);
        
		String mode =
			rundata.getParameters().getString(SecurityConstants.PARAM_MODE);

		if (mode != null
			&& (mode.equals(SecurityConstants.PARAM_MODE_DELETE)
				|| mode.equals(SecurityConstants.PARAM_MODE_UPDATE)))
		{
			String clientName =
				rundata.getParameters().getString("client_name");
			ClientEntry clientEntry =
				(ClientEntry) Registry.getEntry(Registry.CLIENT, clientName);
			context.put("entry", clientEntry);
		}
	}

	/**
	 * Add a mimetype to a client entry
	 * @param rundata The turbine rundata context for this request.
	 * @param context The velocity context for this request.
	 * @throws Exception
	 */
	public void doAddmimetype(RunData rundata, Context context)
		throws Exception
	{
		try
		{
			String clientName =
				rundata.getParameters().getString("client_name");
			ClientEntry clientEntry =
				(ClientEntry) Registry.getEntry(Registry.CLIENT, clientName);
			if (clientEntry != null)
			{
                String mimeType = rundata.getParameters().getString("mime_type");
				clientEntry.getMimetypeMap().addMimetype(mimeType);

				Registry.addEntry(Registry.CLIENT, clientEntry);
				clearUserData(rundata);
			}
			else
			{
                DynamicURI duri = redirect(rundata, SecurityConstants.PARAM_MODE_UPDATE, SecurityConstants.MID_INVALID_ENTITY_NAME);
				rundata.setRedirectURI(duri.toString());
				resetForm(rundata);
                
                logger.error(this.getClass().getName() + ": Failed to find registry entry while trying to add mime type");
			}
		}
		catch (Exception e)
		{
            DynamicURI duri = redirect(rundata, SecurityConstants.PARAM_MODE_UPDATE, SecurityConstants.MID_UPDATE_FAILED);
			rundata.setRedirectURI(duri.toString());
			resetForm(rundata);
            
            logger.error("Error addin mime type", e);
		}
	}
    

	/**
	 * Remove mime types from a client entry
	 * @param rundata The turbine rundata context for this request.
	 * @param context The velocity context for this request.
	 * @throws Exception
	 */
    
	public void doRemovemimetype(RunData rundata, Context context)
		throws Exception
	{
		try
		{
			String clientName =
				rundata.getParameters().getString("client_name");
			ClientEntry clientEntry =
				(ClientEntry) Registry.getEntry(Registry.CLIENT, clientName);
			if (clientEntry != null)
			{
				String[] mimeTypes =
                    rundata.getParameters().getStrings("mime_type");
				if (mimeTypes != null && mimeTypes.length > 0)
				{
					for (int i = 0; i < mimeTypes.length; i++)
					{
						String mimeType = mimeTypes[i];

						clientEntry.getMimetypeMap().removeMimetype(mimeType);
					}

					Registry.addEntry(Registry.CLIENT, clientEntry);
					clearUserData(rundata);
				}
			}
			else
			{
                DynamicURI duri = redirect(rundata, SecurityConstants.PARAM_MODE_UPDATE, SecurityConstants.MID_INVALID_ENTITY_NAME);
				rundata.setRedirectURI(duri.toString());
				resetForm(rundata);
                
                logger.error(this.getClass().getName() + ": Failed to find registry entry while trying to remove mime types");
			}
		}
		catch (Exception e)
		{
            DynamicURI duri = redirect(rundata, SecurityConstants.PARAM_MODE_UPDATE, SecurityConstants.MID_UPDATE_FAILED);
			rundata.setRedirectURI(duri.toString());
			resetForm(rundata);
            
            logger.error("Error removing mime types", e);
		}
	}
    

	/**
     * @see org.apache.jetspeed.modules.actions.portlets.RegistryUpdateAction#updateRegistryEntry(org.apache.turbine.util.RunData, org.apache.jetspeed.om.registry.RegistryEntry)
     */
    protected void updateRegistryEntry(RunData rundata, RegistryEntry registryEntry) throws Exception
	{
        super.updateRegistryEntry(rundata, registryEntry);
        
        updateClientEntry(rundata, (ClientEntry)registryEntry);
	}

	/**
	 * Set the client entry parameters from the input parameters
	 * @param rundata The turbine rundata context for this request.
	 * @param context The velocity context for this request.
	 */
	private void updateClientEntry(
		RunData rundata,
        ClientEntry clientEntry) throws Exception
	{
		String userAgentPattern =
			rundata.getParameters().getString("user_agent_pattern");
		String manufacturer = rundata.getParameters().getString("manufacturer");
		String model = rundata.getParameters().getString("model");
		String version = rundata.getParameters().getString("version");

        if(hasChanged(clientEntry.getUseragentpattern(), userAgentPattern))
		{
            try
			{
                RE re = new RE(userAgentPattern);
                clientEntry.setUseragentpattern(userAgentPattern);
			}
            catch(RESyntaxException e)
			{
                logger.error(whoAmI + "Illegal regular expression syntax " + userAgentPattern + " for user agent");
                logger.debug(whoAmI + "Illegal regular expression syntax for user agent", e);
                
                throw new IllegalArgumentException("Illegal regular expression syntax for user agent");
			}
		}
        if(hasChanged(clientEntry.getManufacturer(), manufacturer))
		{
            clientEntry.setManufacturer(manufacturer);
        }
        if(hasChanged(clientEntry.getModel(), model))
        {
            clientEntry.setModel(model);
        }
        if(hasChanged(clientEntry.getVersion(), version))
        {
            clientEntry.setVersion(version);
		}
	}

	/**
	 * Populates the user's temp storage with form data
	 * @param rundata The turbine rundata context for this request.
	 */
    protected void resetForm(RunData rundata)
	{
        super.resetForm(rundata);
        
		String userAgentPattern =
			rundata.getParameters().getString("user_agent_pattern");
		String manufacturer = rundata.getParameters().getString("manufacturer");
		String model = rundata.getParameters().getString("model");
		String version = rundata.getParameters().getString("version");

		String capability = rundata.getParameters().getString("capability");
        String mimeType = rundata.getParameters().getString("mime_type");

		rundata.getUser().setTemp("user_agent_pattern", userAgentPattern);
		rundata.getUser().setTemp("manufacturer", manufacturer);
		rundata.getUser().setTemp("model", model);
		rundata.getUser().setTemp("version", version);

		rundata.getUser().setTemp("capability", capability);
		rundata.getUser().setTemp("mimetype", mimeType);
	}

	/**
	 * Clears the temporary storage of any data that was used
	 * @param rundata The turbine rundata context for this request.
	 */
    protected void clearUserData(RunData rundata)
    {
	try
	{
            super.clearUserData(rundata);
            
            rundata.getUser().removeTemp("user_agent_pattern");
            rundata.getUser().removeTemp("manufacturer");
            rundata.getUser().removeTemp("model");
            rundata.getUser().removeTemp("version");

            rundata.getUser().removeTemp("capability");
            rundata.getUser().removeTemp("mime_type");
        }
        catch (Exception e)
        {
                   
            if (logger.isDebugEnabled())
            {
                logger.debug("ClientUpdateAction: Failed to clear user data");
            }
        }
    }
}
