package jp.sourceforge.jetspeedjapan.portlet.weblog.om;

/* ====================================================================================
 * Copyright (c) 2003, jetspeed-japan.sourceforge.jp
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *     * Redistributions of source code must retain the above copyright notice,
 *     * this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *     * notice, this list of conditions and the following disclaimer in the
 *     * documentation and/or other materials provided with the distribution.
 *     * Neither the name of the jetspeed-japan.sourceforge.jp nor the names 
 *       of its contributors
 *     * may be used to endorse or promote products derived from this software
 *     * without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================================
 */

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.torque.Torque;
import org.apache.torque.TorqueException;
import org.apache.torque.util.Criteria;

import jp.sourceforge.jetspeedjapan.portlet.weblog.WebLogException;
import jp.sourceforge.jetspeedjapan.portlet.weblog.om.utils.PageList;

/** 
 * This is the Peer class for Weblog object model.
 * 
 * @author <a href="mailto:shinsuke@users.sourceforge.jp">Shinsuke Sugaya</a>
 */
public class WeblogPeer extends jp.sourceforge.jetspeedjapan.portlet.weblog.om.BaseWeblogPeer
{
    /**
     * Static initialization of the logger for this class
     */
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(WeblogPeer.class.getName());

    public static Weblog add(Connection con, String owner, int year, int month, int day) throws WebLogException
    {
        // Insert Weblog
        try
        {
            Weblog wlog = new Weblog();
            wlog.setOwner(owner);
            wlog.setYear(year);
            wlog.setMonth(month);
            wlog.setDay(day);
            // TODO: default Title message should be formatted.
            wlog.setTitle(Integer.toString(year) + "/" + Integer.toString(month) + "/" + Integer.toString(day));
            wlog.save(con);
            return wlog;
        }
        catch (TorqueException e)
        {
            logger.warn("add: " + e.toString());
            throw new WebLogException("Cannot create Weblog data.", e);
        }

    }

    /**
     * Gets Weblog data for specified value.
     * 
     * @param owner Owner name. null is a invalid value.
     * @param year  -1 is a invalid value.
     * @param month -1 is a invalid value.
     * @param day   -1 is a invalid value.
     * @return
     * @throws WebLogException
     */
    public static List getWeblogs(String owner, int year, int month, int day) throws WebLogException
    {
        Criteria crit = new Criteria();
        if (owner != null)
        {
            crit.add(WeblogPeer.OWNER, owner);
        }
        if (year > 0)
        {
            crit.add(WeblogPeer.YEAR, year);
        }
        if (month > 0)
        {
            crit.add(WeblogPeer.MONTH, month);
        }
        if (day > 0)
        {
            crit.add(WeblogPeer.DAY, day);
        }

        try
        {
            return WeblogPeer.doSelect(crit);
        }
        catch (TorqueException e)
        {
            logger.warn("getWeblogs: " + e.toString());
            throw new WebLogException("Cannot get Weblog data.", e);
        }

    }

    public static PageList getWeblogs(Connection con, String owner, int year, int month, int day, int page, int size)
        throws WebLogException
    {
        Criteria crit = new Criteria();
        if (owner != null)
        {
            crit.add(WeblogPeer.OWNER, owner);
        }
        if (year != -1)
        {
            crit.add(WeblogPeer.YEAR, year);
        }
        if (month > 0 && month < 13)
        {
            crit.add(WeblogPeer.MONTH, month);
        }
        if (day > 0 && day < 32)
        {
            crit.add(WeblogPeer.DAY, day);
        }

        crit.addDescendingOrderByColumn(WeblogPeer.YEAR);
        crit.addDescendingOrderByColumn(WeblogPeer.MONTH);
        crit.addDescendingOrderByColumn(WeblogPeer.DAY);

        return WeblogPeer.doPageSelect(con, crit, page, size);
    }
    /**
     * Select items by the given page
     * 
     * @param con
     * @param crit
     * @param page
     * @param size
     * @return
     * @throws TorqueException
     * @throws SQLException
     */
    public static PageList doPageSelect(Connection con, Criteria crit, int page, int size) throws WebLogException
    {
        PageList pageList = new PageList(page, size);
        if (page > 1)
        {
            pageList.setPrevious(true);
        }

        // Set DB name
        if (crit.getDbName() == Torque.getDefaultDB())
        {
            crit.setDbName(DATABASE_NAME);
        }

        List list = null;
        try
        {
            String sql = WeblogPeer.createQueryString(crit);
            sql = StringUtils.replace(sql, "SELECT  ", "SELECT * ");
            sql = StringUtils.replace(sql, "FROM  ", "FROM " + WeblogPeer.TABLE_NAME + " ");

            Statement stat = con.createStatement();
            ResultSet rs = stat.executeQuery(sql);

            int count = 1;
            List pkList = new LinkedList();
            while (rs.next())
            {
                if (count <= page * size)
                {
                    if (count > (page - 1) * size)
                    {
                        pkList.add(rs.getString(1));
                    }
                }
                else
                {
                    pageList.setNext(true);
                }
                count++;
            }

            // set Total Size
            pageList.setTotalSize(count);

            rs.close();
            stat.close();

            if (pkList.isEmpty())
            {
                return pageList;
            }
            crit.andIn(WeblogPeer.WEBLOG_ID, pkList);
            pageList.setList(WeblogPeer.doSelect(crit, con));
        }
        catch (TorqueException e)
        {
            logger.warn("doSelectByPage: " + e.toString());
            throw new WebLogException("Could not find Weblog data. ", e);
        }
        catch (SQLException e)
        {
            logger.warn("doSelectByPage: " + e.toString());
            throw new WebLogException("Could not find Weblog data. ", e);
        }

        return pageList;
    }

    /**
     * Gets Web Log message for the given date.
     */
    public static Weblog getInstance(Connection con, String owner, int year, int month, int day) throws WebLogException
    {
        if (owner == null || year == -1 || month < 1 || month > 12 || day < 1 || day > 31)
        {
            throw new WebLogException("Invalid Weblog data.");
        }
        
        Criteria crit = new Criteria();
        crit.add(WeblogPeer.OWNER, owner);
        crit.add(WeblogPeer.YEAR, year);
        crit.add(WeblogPeer.MONTH, month);
        crit.add(WeblogPeer.DAY, day);

        List l;
        try
        {
            l = WeblogPeer.doSelect(crit, con);
        }
        catch (TorqueException e)
        {
            logger.warn("getInstance: " + e.toString());
            throw new WebLogException("Could not get Weblog data. (Error Code: TorqueException on Transaction)", e);
        }

        if (!l.isEmpty() && l.get(0) instanceof Weblog)
        {
            return (Weblog) (l.get(0));
        }
        return null;

    }

    /**
     * Gets the latest Web Log data.
     * 
     * @param con
     * @param owner
     * @param page
     * @param size
     * @return
     * @throws WebLogException
     */
    public static PageList getLatestWeblogs(Connection con, String owner, int page, int size) throws WebLogException
    {
        Criteria crit = new Criteria();
        if (owner != null)
        {
            crit.add(WeblogPeer.OWNER, owner);
        }
        crit.addDescendingOrderByColumn(WeblogPeer.YEAR);
        crit.addDescendingOrderByColumn(WeblogPeer.MONTH);
        crit.addDescendingOrderByColumn(WeblogPeer.DAY);

        return WeblogPeer.doPageSelect(con, crit, page, size);

    }
}
