/**********************************************************************************
*
* $Header: /home/cvspublic/jakarta-jetspeed/src/java/org/apache/jetspeed/services/statemanager/JetspeedStateManagerService.java,v 1.5 2002/06/10 19:57:18 ggolden Exp $
*
* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// package
package org.apache.jetspeed.services.statemanager;

// imports
import java.util.HashMap;
import java.util.Map;
import java.util.Collections;
import java.util.Iterator;
import java.util.Vector;
import java.util.Set;

import org.apache.jetspeed.services.statemanager.BaseStateManagerService;

/**
* <p>JetspeedStateManagerService is an implementation of the BaseStateManagerService
* which manages the states stored in a local Map (synchronized HashMap).</p>
* <p>Note: This implementation of the StateManagerService is independent of all other
* services; but it has no automatic way to retire no longer used state.  If the
* application does not explicitly retire the states created, they will hang around
* forever. (see clear() and retireState() of the StateManagerService).</p>
* @version $Revision: 1.5 $
* @see org.apache.jetspeed.services.statemanager.BaseStateManagerService
* @see org.apache.jetspeed.services.statemanager.StateManagerService
* @see org.apache.jetspeed.services.statemanager.SessionState
* @author <a href="mailto:ggolden@apache.org">Glenn R. Golden</a>
*/
public class JetspeedStateManagerService
    extends BaseStateManagerService
{
    /** Store each set of state parameters by state key
        (each is a HashMap keyed by parameter name) */
    private Map m_states = null;

    /**
    * Initialize the states storage.
    */
    protected void initStates()
    {
        // create our states map synchronized
        m_states = Collections.synchronizedMap(new HashMap());

    }   // initStates

    /**
    * Cleanup the states storage.
    */
    protected void shutdownStates()
    {
        m_states.clear();
        m_states = null;

    }   // shutdownStates

    /**
    * Access the Map which is the set of attributes for a state.
    * @param key The state key.
    * @return The Map which is the set of attributes for a state.
    */
    protected Map getState( String key )
    {
        return (Map) m_states.get(key);

    }   // getState

    /**
    * Add a new state to the states we are managing.
    * @param key The state key.
    * @param state The Map which is the set of attributes for the state.
    */
    protected void addState( String key, Map state )
    {
        m_states.put(key, state);

    }   // addState

    /**
    * Remove a state from the states we are managing.
    * @param key The state key.
    */
    protected void removeState( String key )
    {
        m_states.remove(key);

    }   // removeState

    /**
    * Access an array of the keys of all states managed, those that start with the parameter.
    * @param start The starting string used to select the keys.
    * @return an array of the keys of all states managed.
    */
    protected String[] getStateKeys( String start )
    {
        // collect for return
        Vector rv = new Vector();

        // get the entire set of keys to iterate over
        Set allStateKeys = m_states.keySet();
        synchronized (m_states)
        {
            Iterator i = allStateKeys.iterator();
            while (i.hasNext())
            {
                String key = (String) i.next();
                
                // if this matches our pattern
                if (key.startsWith(start))
                {
                    rv.add(key);
                }
            }
        }

        if (rv.size() == 0) return null;

        return (String[]) rv.toArray(new String[rv.size()]);

    }   // getStateKeys

}   // JetspeedStateManagerService

/**********************************************************************************
*
* $Header: /home/cvspublic/jakarta-jetspeed/src/java/org/apache/jetspeed/services/statemanager/JetspeedStateManagerService.java,v 1.5 2002/06/10 19:57:18 ggolden Exp $
*
**********************************************************************************/

