/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security;

import java.util.Iterator;
import java.util.HashMap;

// Junit imports
import junit.framework.Test;
import junit.framework.TestSuite;
import junit.framework.TestCase;


import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;


import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.om.security.Permission;
import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.om.security.JetspeedRoleFactory;

/**
 * Unit test for SecurityCacheService
 * 
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: TestSecurityCache.java,v 1.3 2002/10/15 17:35:33 taylor Exp $
 */

public class TestSecurityCache extends TestCase {    

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestSecurityCache( String name ) {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[]) 
    {
        junit.awtui.TestRunner.main( new String[] { TestSecurityCache.class.getName() } );
    }
 
    public void setup() 
    {
        //System.out.println("Setup: Testing Turbine Role Management");         
    }

    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite() 
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestSecurityCache.class );
    }

    /**
     * Tests getRoles method
     * @throws Exception
     */

    public void testLoadCache() throws Exception 
    {
        SecurityCacheService service = getService();

        try
        {
            JetspeedUser user = JetspeedSecurity.getUser("turbine");
            service.load(user.getUserName());
            Role role = service.getRole(user.getUserName(), "user");
            assertTrue(role.getName().equals("user"));
            assertTrue(service.hasRole(user.getUserName(), "user"));
            assertTrue(service.hasPermission("user", "view"));
            assertTrue(service.hasPermission("user", "customize"));
            assertTrue(service.hasPermission("user", "maximize"));
            assertTrue(!service.hasPermission("user", "failure"));
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed loadCache Test OK ");

    }


    public void testAddRemoveFromCache() throws Exception 
    {
        SecurityCacheService service = getService();

        try
        {
            JetspeedUser user = JetspeedSecurity.getUser("anon");
            service.load(user.getUserName());

            Role role1 = service.getRole(user.getUserName(), "guest");
            assertTrue(role1.getName().equals("guest"));
            assertTrue(service.hasPermission("guest", "view"));

            // add role
            Role role2 = JetspeedSecurity.getRole("user");
            service.addRole(user.getUserName(), role2);
            assertTrue(service.hasRole(user.getUserName(), "user"));
            assertTrue(service.getRole(user.getUserName(),"user").getName().equals("user"));

            // remove role
            service.removeRole(user.getUserName(), "user");
            assertTrue(!service.hasRole(user.getUserName(), "user"));
            Role role3 = service.getRole(user.getUserName(),"user");
            assertTrue(null == role3);

            // add permission
            Permission perm1 = JetspeedSecurity.getPermission("detach");
            assertTrue(null != perm1);
            service.addPermission("guest", perm1);
            Permission permission = service.getPermission("guest", "detach");
            assertTrue(permission.getName().equals("detach"));
            assertTrue(service.hasPermission("guest", "detach"));

            // remove permission
            service.removePermission("guest", "detach");
            assertTrue(!service.hasPermission( "guest", "detach"));
            Permission perm2 = service.getPermission( "guest", "detach");
            assertTrue(null == perm2);

        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed addRemoveFromCache Test OK ");

    }

    public void testRemoveAll() throws Exception 
    {
        SecurityCacheService service = getService();
        try
        {
            Role role = JetspeedSecurity.getRole("admin");
            JetspeedUser anon = JetspeedSecurity.getUser("anon");
            service.load(anon.getUserName());
            JetspeedUser turbine = JetspeedSecurity.getUser("turbine");
            service.load(turbine.getUserName());

            service.addRole(anon.getUserName(), role);
            service.addRole(turbine.getUserName(), role);

            assertTrue(service.hasRole(anon.getUserName(), role.getName()));
            assertTrue(service.hasRole(turbine.getUserName(), role.getName()));

            service.removeAllRoles("admin");

            assertTrue(!service.hasRole(anon.getUserName(), role.getName()));
            assertTrue(!service.hasRole(turbine.getUserName(), role.getName()));

        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed removeAll Test OK ");

    }

  /*
    Configuration object to run Turbine outside a servlet container
    ( uses turbine.properties )
    */
    private static TurbineConfig config = null;
    
    /**
    Sets up TurbineConfig using the system property:
    <pre>turbine.properties</pre>
    */
    static
    {
        try
        {
            config = new TurbineConfig( "../webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }

    private static SecurityCacheService getService()
    {
        return (SecurityCacheService)TurbineServices
                .getInstance()
                .getService(SecurityCacheService.SERVICE_NAME);
    }

}






