/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
 
package org.apache.jetspeed.services.psmlmanager;

import java.util.Iterator;

import org.apache.turbine.services.TurbineServices;

import org.apache.jetspeed.services.psmlmanager.db.DatabasePsmlManagerService;

// Jetspeed Security service
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.security.JetspeedSecurityException;
import org.apache.jetspeed.services.security.UnknownUserException;
import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.om.security.Group;

// Profile and ProfileLocator interface 
import org.apache.jetspeed.om.profile.Profile;
import org.apache.jetspeed.om.profile.ProfileLocator;
import org.apache.jetspeed.services.PsmlManager;
import org.apache.jetspeed.om.profile.QueryLocator;
import org.apache.turbine.services.InitializationException;
import org.apache.turbine.util.TurbineConfig;

import org.apache.turbine.util.Log;

/**
 * Reads all PSML files from the file system and imports them into PSML DB
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: PsmlImporter.java,v 1.13 2002/07/29 02:25:27 taylor Exp $
 */
public class PsmlImporter
{   
    protected boolean check = true;

    public PsmlImporter()
    {
    }

    public static void main(String args[]) 
    {
        System.out.println("***** PSML Importer *****");
        boolean checkImport = true;
        
        //
        // initialize and bootstrap services
        //
        try
        {
            String root = "./webapp";
            String properties = "/WEB-INF/conf/TurbineResources.properties";
            if (args.length > 0)
            {
                if (args[0].equalsIgnoreCase("true"))
                    checkImport = true;
                else
                    checkImport = false;
            }
            if (args.length > 1)
            {
                root = args[1];
            }
            if (args.length > 2)
            {
                properties = args[2];
            }
            TurbineConfig config = new TurbineConfig( root, properties);
            config.init();
        }
        catch (Exception e)
        {
            String msg = "PSML Importer: error initializing Turbine configuration";
            Log.error(msg, e);
            System.out.println(msg);
            e.printStackTrace();
            System.exit(0);
        }

        //
        // get a handle to the exporter service
        //
        PsmlManagerService exporterService = null;
        PsmlManagerService importerService = null;

        try
        {
            exporterService = (PsmlManagerService)TurbineServices.getInstance().getService("PsmlImportManager");
        }
        catch (org.apache.turbine.services.InstantiationException e)
        {
            String msg = "PSML Importer: error loading Psml Exporter Service";
            Log.error(msg, e);
            System.out.println(msg);
            e.printStackTrace();
            System.exit(0);
        }

        //
        // get a handle to the importer service
        //
        try
        {
            importerService = PsmlManager.getService();
        }
        catch (org.apache.turbine.services.InstantiationException e)
        {
            String msg = "PSML Importer: error loading Psml Importer Service";
            Log.error(msg, e);
            System.out.println(msg);
            e.printStackTrace();
            System.exit(0);
        }

        if (exporterService.getClass().getName().equals(importerService.getClass().getName()))
        {
            String msg = "PSML Importer Error: Importer Class cannot equal Exporter Class.";
            Log.error(msg);
            System.out.println(msg);
            System.exit(0);
        }

        PsmlImporter importer = new PsmlImporter();
        importer.setCheck(checkImport);
        boolean ran = importer.run(exporterService, importerService);

        if (ran)
        {
            System.out.println("**** PSML Importer - completed");
        }        

        System.exit(1);

    }

    public boolean run(PsmlManagerService exporterService,
                    PsmlManagerService importerService)
    {
        String msg;
        int count = 0;
        try
        {
           if (check && alreadyImported())
                return false; 

            msg = "Running with Importer Service: " + importerService.getClass();
            System.out.println(msg);
            Log.info(msg);

            msg = "Running with Exporter Service: " + exporterService.getClass();
            System.out.println(msg);
            Log.info(msg);


            QueryLocator locator = new QueryLocator(QueryLocator.QUERY_ALL);
            count = exporterService.export(importerService, locator);
        }
        catch (Exception e)
        {
            System.out.println("Error importing: " + e.toString());
            Log.error("Error importing: " , e);
            e.printStackTrace();
            return false;
        }             
        msg = "PSMLImporter completed. Exported " + count + " profiles";
        System.out.println(msg);
        Log.info(msg);
        return true;
    }


    /*
     * Check to see if import has already completed.
     * Only considers a "onetime" import, checking for the "admin" user.
     *
     * @return true if import was already ran.
     */
    public boolean alreadyImported() 
    {
        try 
        {
            JetspeedUser user = JetspeedSecurity.getUser("admin");
            QueryLocator ql = new QueryLocator(QueryLocator.QUERY_USER);
            ql.setUser(user);
            Iterator iterator = PsmlManager.query(ql);
            if (iterator.hasNext())
            {                      
                String msg = "PSMLImporter: Detected database is populated. No need to import.";
                System.out.println(msg);
                Log.info(msg);
                return true;    // record found
            }
            return false;   // record not found
        }
        catch (UnknownUserException e)
        {
            return false;  // record not found
        }
        catch (JetspeedSecurityException e)
        {
            String msg = "Failed to run import: Database Access Error detecting database on import: ";
            Log.error(msg, e);    
            System.out.println(msg + e.toString());
            return true;
        }
    }

    public void setCheck(boolean check)
    {
        this.check = check;
    }

    public boolean getCheck()
    {
        return this.check;
    }

}
