/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services;

import java.util.Iterator;
import java.security.Principal;

import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.Log;
import org.apache.turbine.services.rundata.RunDataService;

import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.om.security.Group;
import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.om.security.Permission;
import org.apache.jetspeed.om.profile.Profile;
import org.apache.jetspeed.om.profile.BaseProfile;
import org.apache.jetspeed.services.Profiler;
import org.apache.jetspeed.services.PsmlManager;
import org.apache.jetspeed.services.rundata.JetspeedRunDataService;
import org.apache.jetspeed.services.rundata.JetspeedRunData;
import org.apache.jetspeed.portal.Portlet;

import org.apache.jetspeed.om.profile.Entry;
import org.apache.jetspeed.om.profile.ProfileException;
import org.apache.jetspeed.om.registry.RegistryEntry;
import org.apache.jetspeed.om.security.JetspeedUserFactory;
import org.apache.jetspeed.om.security.UserNamePrincipal;

import org.apache.jetspeed.services.JetspeedPortalAccessController;
import org.apache.jetspeed.services.JetspeedUserManagement;
import org.apache.jetspeed.services.JetspeedAuthentication;
import org.apache.jetspeed.services.security.JetspeedRoleManagement;
import org.apache.jetspeed.services.security.RoleManagement;
import org.apache.jetspeed.services.security.JetspeedGroupManagement;
import org.apache.jetspeed.services.security.GroupManagement;
import org.apache.jetspeed.services.security.JetspeedPermissionManagement;
import org.apache.jetspeed.services.security.PermissionManagement;

import org.apache.jetspeed.services.security.JetspeedSecurityService;
import org.apache.jetspeed.services.security.JetspeedSecurityException;
import org.apache.jetspeed.services.security.PortalResource;
import org.apache.jetspeed.services.security.LoginException;


/**
 * <P>This is a commodity static accessor class around the 
 * <code>JetspeedSecurityService</code></P>
 * 
 * @see org.apache.jetspeed.services.security.JetspeedSecurityService
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @author <a href="mailto:morciuch@apache.org">Mark Orciuch</a> 
 * @version $Id: JetspeedSecurity.java,v 1.13 2002/09/10 15:17:29 morciuch Exp $
 */

abstract public class JetspeedSecurity /* extends TurbineSecurity */
{
    public static final String PERMISSION_VIEW = "view";
    public static final String PERMISSION_CUSTOMIZE = "customize";
    public static final String PERMISSION_MAXIMIZE = "maximize";
    public static final String PERMISSION_MINIMIZE = "minimize";
    public static final String PERMISSION_PERSONALIZE = "personalize";
    public static final String PERMISSION_DETACH = "detach";
    public static final String PERMISSION_CLOSE = "close";
    public static final String PERMISSION_INFO = "info";
    public static final String PERMISSION_PRINT_FRIENDLY = "print_friendly";

    // Jetspeed security only has one group. 
    // Access Control checks are only role-based.
    // If a user has the specified role for the resource, then the user can access that resource
    public static final String JETSPEED_GROUP = "Jetspeed";
    public static final String JETSPEED_GROUP_ID = "1";

    public static final String JETSPEED_ROLE_USER = "user";
    public static final String JETSPEED_ROLE_ADMIN = "admin";

    /** 
     * Commodity method for getting a reference to the service
     * singleton
     */
    public static JetspeedSecurityService getService()
    {
        return (JetspeedSecurityService)TurbineServices
                .getInstance()
                .getService(JetspeedSecurityService.SERVICE_NAME);     
    }

    //////////////////////////////////////////////////////////////////////////
    // PortalAuthentication
    /////////////////////////////////////////////////////////////////////////

    /**
     * @see PortalAuthentication#login
     */
    public static JetspeedUser login(String username, String password)
        throws LoginException
    {
        return JetspeedAuthentication.login(username, password);
    }

    /**
     * @see PortalAuthentication#getAnonymousUser
     */
    public static JetspeedUser getAnonymousUser()
        throws LoginException
    {
        return JetspeedAuthentication.getAnonymousUser();
    }

    /**
     * @see PortalAuthentication#logout
     */
    public static void logout()
        throws LoginException
    {
        JetspeedAuthentication.logout();
    }


    //////////////////////////////////////////////////////////////////////////
    // PortalAuthorization
    /////////////////////////////////////////////////////////////////////////

    /**
     * @see JetspeedPortalAccessController#checkPermission
     */
    public static boolean checkPermission(JetspeedUser user, Entry entry, String action)
    {
        return JetspeedPortalAccessController.checkPermission(user, entry, action);
    }

    /**
     * @see JetspeedPortalAccessController#checkPermission
     */
    public static boolean checkPermission(JetspeedUser user, Portlet portlet, String action)
    {
        return JetspeedPortalAccessController.checkPermission(user, portlet, action);
    }

    /**
     * @see JetspeedPortalAccessController#checkPermission
     */
    public static boolean checkPermission(JetspeedUser user, PortalResource resource, String action)
    {
        return JetspeedPortalAccessController.checkPermission(user, resource, action);
    }


    //////////////////////////////////////////////////////////////////////////
    // UserManagement
    /////////////////////////////////////////////////////////////////////////

    /**
     * @see UserManagement#getUser
     */
    public static JetspeedUser getUser(Principal principal) 
        throws JetspeedSecurityException
    {
        return JetspeedUserManagement.getUser(principal);
    }

    /**
     * @see UserManagement#getUsers
     */
    public static Iterator getUsers()
        throws JetspeedSecurityException
    {
        return JetspeedUserManagement.getUsers();
    }

    /**
     * @see UserManagement#saveUser
     */
    public static void saveUser( JetspeedUser user )
        throws JetspeedSecurityException
    {
        JetspeedUserManagement.saveUser(user);
    }

    /**
     * @see UserManagement#addUser
     */
    public static void addUser( JetspeedUser user )
        throws JetspeedSecurityException
    {
        JetspeedUserManagement.addUser(user);
    }

    /**
     * @see UserManagement#getUsers(String)
     */
    public static Iterator getUsers(String filter)
        throws JetspeedSecurityException
    {
        return JetspeedUserManagement.getUsers(filter);
    }

    /**
     * @see UserManagement#removeUser
     */    
    public static void removeUser(Principal principal)
        throws JetspeedSecurityException
    {
        JetspeedUserManagement.removeUser(principal);
    }

    /**
     * @see UserManagement#getUser
     */    
    public static JetspeedUser getUser(String username) 
        throws JetspeedSecurityException
    {
        return JetspeedUserManagement.getUser(new UserNamePrincipal(username));
    }

    /**
     * @see UserManagement#removeUser
     */    
    public static void removeUser(String username)
        throws JetspeedSecurityException
    {
        JetspeedUserManagement.removeUser(new UserNamePrincipal(username));
    }


    //////////////////////////////////////////////////////////////////////////
    // CredentialsManagement
    /////////////////////////////////////////////////////////////////////////

    /**
     * @see CredentialsManagement#changePassword
     */
    public static void changePassword( JetspeedUser user,
                         String oldPassword, 
                         String newPassword )
        throws JetspeedSecurityException
    {
        JetspeedUserManagement.changePassword(user, oldPassword, newPassword);
    
    }

    /**
     * @see CredentialsManagement#forcePassword
     */
    public static void forcePassword( JetspeedUser user, String password )
        throws JetspeedSecurityException
    {
        JetspeedUserManagement.forcePassword(user, password);
    }

    /**
     * @see CredentialsManagement#encryptPassword
     */
    public static String encryptPassword( String password )
        throws JetspeedSecurityException
    {
        return JetspeedUserManagement.encryptPassword(password);
    }
     

    //////////////////////////////////////////////////////////////////////////
    // Role Management
    /////////////////////////////////////////////////////////////////////////


    /**
     * @see RoleManagement#getRoles(String)
     */    
    public static Iterator getRoles(String username)
        throws JetspeedSecurityException
    {
        return JetspeedRoleManagement.getRoles(username);
    }

    /**
     * @see RoleManagement#getRoles
     */    
    public static Iterator getRoles()
        throws JetspeedSecurityException
    {
        return JetspeedRoleManagement.getRoles();
    }

    /**
     * @see RoleManagement#addRole
     */    
    public static void addRole(Role role)
        throws JetspeedSecurityException
    {
        JetspeedRoleManagement.addRole(role);
    }

    /**
     * @see RoleManagement#saveRole
     */    
    public static void saveRole(Role role)
        throws JetspeedSecurityException
    {
        JetspeedRoleManagement.saveRole(role);
    }

    /**
     * @see RoleManagement#removeRole
     */    
    public static void removeRole(String rolename)
        throws JetspeedSecurityException
    {
        JetspeedRoleManagement.removeRole(rolename);
    }

    /**
     * @see RoleManagement#grantRole
     */    
    public static void grantRole(String username, String rolename)
        throws JetspeedSecurityException
    {
        JetspeedRoleManagement.grantRole(username,rolename);
    }

    /**
     * @see RoleManagement#revokeRole
     */    
    public static void revokeRole(String username, String rolename)
        throws JetspeedSecurityException
    {
        JetspeedRoleManagement.revokeRole(username,rolename);
    }

    /**
     * @see RoleManagement#hasRole
     */    
    public static boolean hasRole(String username, String rolename)
        throws JetspeedSecurityException
    {
        return JetspeedRoleManagement.hasRole(username,rolename);
    }

    /**
     * @see RoleManagement#getRole
     */    
    public static Role getRole(String rolename)
        throws JetspeedSecurityException
    {
        return JetspeedRoleManagement.getRole(rolename);
    }
        

    //////////////////////////////////////////////////////////////////////////
    // Group Management
    /////////////////////////////////////////////////////////////////////////


    /**
     * @see GroupManagement#getGroups(String)
     */    
    public static Iterator getGroups(String username)
        throws JetspeedSecurityException
    {
        return JetspeedGroupManagement.getGroups(username);
    }

    /**
     * @see GroupManagement#getGroups
     */    
    public static Iterator getGroups()
        throws JetspeedSecurityException
    {
        return JetspeedGroupManagement.getGroups();
    }

    /**
     * @see GroupManagement#addGroup
     */    
    public static void addGroup(Group group)
        throws JetspeedSecurityException
    {
        JetspeedGroupManagement.addGroup(group);
    }

    /**
     * @see GroupManagement#saveGroup
     */    
    public static void saveGroup(Group group)
        throws JetspeedSecurityException
    {
        JetspeedGroupManagement.saveGroup(group);
    }

    /**
     * @see GroupManagement#removeGroup
     */    
    public static void removeGroup(String groupname)
        throws JetspeedSecurityException
    {
        JetspeedGroupManagement.removeGroup(groupname);
    }

    /**
     * @see GroupManagement#joinGroup
     */    
    public static void joinGroup(String username, String groupname)
        throws JetspeedSecurityException
    {
        JetspeedGroupManagement.joinGroup(username,groupname);
    }

    /**
     * @see GroupManagement#revokeGroup
     */    
    public static void unjoinGroup(String username, String groupname)
        throws JetspeedSecurityException
    {
        JetspeedGroupManagement.unjoinGroup(username,groupname);
    }

    /**
     * @see GroupManagement#inGroup
     */    
    public static boolean inGroup(String username, String groupname)
        throws JetspeedSecurityException
    {
        return JetspeedGroupManagement.inGroup(username,groupname);
    }

    /**
     * @see GroupManagement#getGroup
     */    
    public static Group getGroup(String groupname)
        throws JetspeedSecurityException
    {
        return JetspeedGroupManagement.getGroup(groupname);
    }


    //////////////////////////////////////////////////////////////////////////
    //
    // Required JetspeedSecurity Functions
    //
    // Required Features provided by default JetspeedSecurity
    //
    //////////////////////////////////////////////////////////////////////////

    /**
     * @see JetspeedSecurityService#getUserInstance
     */
    public static JetspeedUser getUserInstance()
    {
        return ((JetspeedSecurityService)getService()).getUserInstance();
    }


    //////////////////////////////////////////////////////////////////////////
    //
    // Optional JetspeedSecurity Features 
    //
    // Features are not required to be implemented by Security Provider
    //
    //////////////////////////////////////////////////////////////////////////

    /**
     * @see JetspeedSecurityService#convertUserName
     */
    public static String convertUserName(String username)
    {
       return ((JetspeedSecurityService)getService()).convertUserName(username);
    }

    /**
     * @see JetspeedSecurityService#convertPassword
     */
    public static String convertPassword(String password)
    {
       return ((JetspeedSecurityService)getService()).convertPassword(password);
    }

    /**
     * @see JetspeedSecurityService#checkDisableAcccount
     */
    public static boolean checkDisableAccount(String username)
    {
        return ((JetspeedSecurityService)getService()).checkDisableAccount(username);
    }

    /**
     * @see JetspeedSecurityService#isDisableCountCheckEnabled
     */
    public static boolean isDisableAccountCheckEnabled()
    {
       return ((JetspeedSecurityService)getService()).isDisableAccountCheckEnabled();
    }

    /**
     * @see JetspeedSecurityService#resetDisableAccountCheck
     */
    public static void resetDisableAccountCheck(String username)
    {
       ((JetspeedSecurityService)getService()).resetDisableAccountCheck(username);
    }

    /**
     * @see JetspeedSecurityService#areActionsDisabledForAnon
     */
    public static boolean areActionsDisabledForAnon()
    {
        return ((JetspeedSecurityService)getService()).areActionsDisabledForAnon();
    }

    /**
     * @see JetspeedSecurityService#areActionsDisabledForAllUsers
     */
    public static boolean areActionsDisabledForAllUsers()
    {
        return ((JetspeedSecurityService)getService()).areActionsDisabledForAllUsers();
    }

    /*
     * @see JetspeedSecurityService#getAnonymousUserName
     */
    public static String getAnonymousUserName()
    {
        return ((JetspeedSecurityService)getService()).getAnonymousUserName();
    }


    //////////////////////////////////////////////////////////////////////////
    //
    // PortalAuthorization - Helpers
    //
    /////////////////////////////////////////////////////////////////////////

    /**
     * @see JetspeedSecurityService#checkPermission
     */
    public static boolean checkPermission(JetspeedRunData runData, String action, Portlet portlet)
    {
        return checkPermission(runData.getJetspeedUser(), portlet, action);
    }

    /**
     * @see JetspeedSecurityService#checkPermission
     */
    public static boolean checkPermission(JetspeedRunData runData, String action, RegistryEntry entry)
    {
      return checkPermission(runData.getJetspeedUser(), new PortalResource(entry), action);
    }

                   
    //////////////////////////////////////////////////////////////////////////
    // Permission Management
    /////////////////////////////////////////////////////////////////////////

    /**
     * @see PermissionManagement#getPermissions(String)
     */
    public static Iterator getPermissions(String rolename)
        throws JetspeedSecurityException
    {
        return JetspeedPermissionManagement.getPermissions(rolename);
    }

    /**
     * @see PermissionManagement#getPermissions
     */
    public static Iterator getPermissions()
        throws JetspeedSecurityException
    {
        return JetspeedPermissionManagement.getPermissions();
    }

    /**
     * @see PermissionManagement#addPermission
     */
    public static void addPermission(Permission permission)
        throws JetspeedSecurityException
    {
        JetspeedPermissionManagement.addPermission(permission);
    }

    /**
     * @see PermissionManagement#savePermission
     */
    public static void savePermission(Permission permission)
        throws JetspeedSecurityException
    {
        JetspeedPermissionManagement.savePermission(permission);
    }

    /**
     * @see PermissionManagement#removePermission
     */
    public static void removePermission(String permissionName)
        throws JetspeedSecurityException
    {
        JetspeedPermissionManagement.removePermission(permissionName);
    }

    /**
     * @see PermissionManagement#grantPermission
     */
    public static void grantPermission(String roleName, String permissionName)
        throws JetspeedSecurityException
    {
        JetspeedPermissionManagement.grantPermission(roleName, permissionName);
    }

    /**
     * @see PermissionManagement#revokePermission
     */
    public static void revokePermission(String roleName, String permissionName)
        throws JetspeedSecurityException
    {
        JetspeedPermissionManagement.revokePermission(roleName, permissionName);
    }

    /**
     * @see PermissionManagement#hasPermission
     */
    public static boolean hasPermission(String roleName, String permissionName)
        throws JetspeedSecurityException
    {
        return JetspeedPermissionManagement.hasPermission(roleName, permissionName);
    }

    /**
     * @see PermissionManagement#getPermission
     */
    public static Permission getPermission(String permissionName)
        throws JetspeedSecurityException
    {
        return JetspeedPermissionManagement.getPermission(permissionName);
    }


}
