/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portal.expire;

//jetspeed stuff
import org.apache.jetspeed.cache.disk.*;
import org.apache.jetspeed.portal.*;
import org.apache.jetspeed.util.*;

//turbine stuff
import org.apache.turbine.util.*;

//java stuff
import java.io.*;
import java.util.*;


/**
A generic class for watching a file and determining if it has changed.

@author <a href="mailto:burton@apache.org">Kevin A. Burton</a>
@version $Id: FileWatcher.java,v 1.12 2002/09/27 02:17:24 taylor Exp $
*/
public class FileWatcher implements Serializable{

    //BEGIN Exception index

    public static final String EXCEPTION_URL_NOT_NULL = 
        "URL can NOT be null here.";
    
    public static final String EXCEPTION_URL_NOT_IN_CACHE = 
        "The URL you specified within the disk cache does not exist: ";
    
    
    //END Exception index    
    
    /**
    The URL on which this FileWatchExpire is based.
    */
    private String url = "";

    /**
    The last time this files URL has been modified on disk.
    */
    private long lastModified = 0;

    /**
    The object that relies on this FileWatcher
    */
    private String parent = "";

    /**
    Create a FileWatcher with no parent info.
    
    @see #FileWatcher( String, String )
    
    @author <a href="mailto:burton@apache.org">Kevin A. Burton</a>
    @version $Id: FileWatcher.java,v 1.12 2002/09/27 02:17:24 taylor Exp $
    */
    public FileWatcher( String url ) throws IOException {
        this( url, null );
    }
                            
    /**
    Create a new FileWatcher to watch the given URL.
    
    @author <a href="mailto:burton@apache.org">Kevin A. Burton</a>
    @version $Id: FileWatcher.java,v 1.12 2002/09/27 02:17:24 taylor Exp $
    */
    public FileWatcher( String url,
                        String parent ) throws IOException {
        
        if ( url == null ) {
            throw new IOException( EXCEPTION_URL_NOT_NULL );
        }

        if ( DiskCacheUtils.isRemote( url ) &&
             DiskCacheUtils.isCached( url ) == false ) {
            
            throw new IOException( EXCEPTION_URL_NOT_IN_CACHE + url );
        }

        //Try to set last modified when creating FileWatcher objet
        try {
            this.lastModified = JetspeedDiskCache.getInstance().getEntry( url ).
                                                            getLastModified();
        } catch (Throwable e) {}
        
        this.url = url;
        this.parent = parent;
    }
    
    /**
    Return true if the URL on which this is based has changed.

    @author <a href="mailto:burton@apache.org">Kevin A. Burton</a>
    @version $Id: FileWatcher.java,v 1.12 2002/09/27 02:17:24 taylor Exp $
    */
    public boolean hasChanged() {

        try {

            //initially set the lastModified data
            if ( this.lastModified == 0 ) {
            
                DiskCacheEntry entry = JetspeedDiskCache.getInstance().getEntry( url );
                
                this.lastModified = entry.getLastModified();

                return false;
            
            }

            //the recent modification... if there was one. otherwise it will 
            // == this.lastModified
            long recent = JetspeedDiskCache.getInstance()
                .getEntry( url ).getLastModified();

            // 0 means always modified
            if ( recent == 0 || 
                 this.lastModified < recent ) {

                String message = "";
                
                if ( this.parent != null ) {
                    message = this.parent + ":  ";
                }
                
                message += "REFRESH: Expiring Portlet because it's URL has been modified on disk -> " + url;
                
                Log.info( message );
                return true;
            } 
            
        } catch ( IOException e ) {
            Log.error( e );
            return false;
        }
        
        //default should be to not expire.  This is set if the URL is null
        return false;        
        
    }
    
}
