/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.modules.parameters; 

// Turbine support
import org.apache.turbine.util.Log;
import org.apache.turbine.util.RunData;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.services.jsp.JspService;

// Java stuff
import java.util.Map;

// jetspeed stuff
import org.apache.jetspeed.services.TemplateLocator;

/**
 * Generic jsp-based presentation style. The following default objects are put in the context:
 * <UL>
 * <LI>data - rundata object</LI>
 * <LI>name - name of the parameter</LI>
 * <LI>value - current value of the parameter</LI>
 * <LI>parms - map of additional style parameters</LI>
 * </UL>
 * 
 * <P>Supporting jsp templates should be placed in ${velocity-templates-root}/parameters folder.</p>
 * 
 * <P>It may be used directly with "template" as the only required parameter. This is useful when the
 * no additional objects are needed by the template.</P>
 * 
 * <P>If additional objects need to be put in the context, a new class extending JspParameterPresentationStyle
 * should be created. Override buildContext to place custom objects in the jsp context.</P>
 * 
 * <P>If "template" parameter is not specified, it is assumed that the template name is "classname.vm".</P>
 * 
 * @author <a href="mailto:morciuch@apache.org">Mark Orciuch</a>
 * @version $Id: JspParameterPresentationStyle.java,v 1.2 2002/11/21 23:49:41 morciuch Exp $
 */

public class JspParameterPresentationStyle extends ParameterPresentationStyle
{

    /**
     * Returns presentation control
     * 
     * @param data - rundata object
     * @param name - parameter name
     * @param value - current parameter value
     * @param parms - additional style parameters
     * @return string
     */
    public String getContent(RunData data, String name, String value, Map parms)
    {
        String result = null;

        // Get reference to jsp service
        JspService jspService = (JspService) TurbineServices.getInstance().getService(JspService.SERVICE_NAME);

        // Put basics in the context
        data.getRequest().setAttribute("data", data);
        data.getRequest().setAttribute("name", name);
        data.getRequest().setAttribute("value", value);
        data.getRequest().setAttribute("parms", parms);
        data.getRequest().setAttribute("events", this.getJavascriptEvents());

        try
        {
            // Add custom objects to the context
            this.buildContext(data, name, value, parms);

            // Build default template name (classname + .vm)
            String className = this.getClass().getName();
            int pos = className.lastIndexOf(".");
            pos = pos < 0 ? 0 : pos + 1;
            className = className.substring(pos);

            // Render the template
            String template = (String) this.getParm("template", className + ".jsp");
            String templatePath = TemplateLocator.locateParameterTemplate(data, template);
            jspService.handleRequest(data, templatePath);
            result = "";
        }
        catch (Exception e)
        {
            Log.error(e);
            // Fallback to input text box presentation style
            result = "<input type=\"text\" name=\"" + name + "\" value=\"" + value + "\"";
        }

        return result;

    }

    /**
     * Override this method to put your own objects in the Velocity context
     * 
     * @param data
     * @param name
     * @param value
     * @param parms
     * @param context
     */
    public void buildContext(RunData data, String name, String value, Map parms)
    {

    }
}