/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
 
package org.apache.jetspeed.modules.actions.controllers;

import org.apache.jetspeed.portal.PortletController;
import org.apache.jetspeed.om.profile.PSMLDocument;
import org.apache.jetspeed.services.rundata.JetspeedRunData;

// Turbine stuff
import org.apache.turbine.util.Log;
import org.apache.turbine.util.RunData;
import org.apache.turbine.services.velocity.TurbineVelocity;
import org.apache.turbine.modules.Action;
import org.apache.turbine.modules.actions.VelocityAction;

// Velocity Stuff
import org.apache.velocity.context.Context;


/**
 * An abstract action class to build VelocityPortlet actions.
 * 
 * <p>Don't call it from the URL, the Portlet and the Action are automatically
 * associated through the registry PortletName
 * 
 * @author <a href="mailto:raphael@apache.org">Raphal Luta</a>
 * @author <a href="mailto:re_carrasco@bco011.sonda.cl">Roberto Carrasco</a>
 */
public abstract class VelocityControllerAction extends VelocityAction
{

    /**
     * This overrides the default Action.perform() to execute the
     * doEvent() method.  If that fails, then it will execute the
     * doPerform() method instead.
     *
     * @param data A Turbine RunData object.
     * @exception Exception, a generic exception.
     */
    protected void perform( RunData rundata )
        throws Exception
    {
        // first try to see if there are some events registered for this
        // action...
        Context context = getContext(rundata);
        if (context != null)
        {
            // if context is already defined, events have already been 
            // processed, call doPerform
            doPerform(rundata);
        }
        else
        {
            context = TurbineVelocity.getContext();
            rundata.getTemplateInfo().setTemplateContext("VelocityActionContext",context);
            try
            {
                executeEvents(rundata, context );
            }                    
            catch (NoSuchMethodException e)
            {
                // no event selected
                doPerform(rundata);
            }
        }
    }

    /**
     * This method is used when you want to short circuit an Action
     * and change the template that will be executed next.
     *
     * @param data Turbine information.
     * @param template The template that will be executed next.
     */
    public void setTemplate(RunData data,
                            String template)
    {
        getContext(data).put( "template" , template );
    }

    /**
     * Return the Context needed by Velocity.
     *
     * @param RunData data
     * @return Context, a context for web pages.
     */
    protected Context getContext(RunData data)
    {
        return (Context)data.getTemplateInfo()
                            .getTemplateContext( "VelocityControllerContext" );
    }

    public void doPerform( RunData rundata, Context context )
    {
        PortletController controller = (PortletController)context.get( "controller" );

        // if we're in customization mode for the given set, handle 
        // customization
        if (((JetspeedRunData)rundata).getMode()==JetspeedRunData.CUSTOMIZE)
        {
            buildCustomizeContext( controller, context, rundata);
            return;
        }

        buildNormalContext( controller, context, rundata);
    }

    /** 
     * Subclasses must override this method to provide default behavior 
     * for the portlet action
     */
    protected void buildCustomizeContext( PortletController controller, 
                                          Context context,
                                          RunData rundata )
    {
        String name = controller.getPortlets().getName();            
        String template = (String)context.get("template");

        int dotIdx = template.lastIndexOf('.');
        if (dotIdx > -1)
        {
            template = template.substring(0,dotIdx)
                       + "-customize.vm";
        }
        else
        {
            template = template+"-customize";
        }
        
        setTemplate(rundata, template);
        
        context.put( "action", controller.getConfig().getInitParameter("action"));
    }

    /** 
     * Subclasses must override this method to provide default behavior 
     * for the portlet action
     */
    protected abstract void buildNormalContext( PortletController controller, 
                                                Context context,
                                                RunData rundata );


    /** Switch out of customize mode
     */
    public void doCancel(RunData data, Context context)
    {
        // nothing to do
    }
}
