/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.daemon.impl;

//jetspeed stuff
import org.apache.jetspeed.daemon.*;
import org.apache.jetspeed.cache.disk.*;
import org.apache.jetspeed.util.*;
import org.apache.jetspeed.services.urlmanager.URLManager;
import org.apache.jetspeed.services.urlmanager.URLFetcher;

//java stuff
import java.io.*;
import java.net.*;
import java.util.Iterator;

//turbine stuff
import org.apache.turbine.util.*;

/**
Manages pulling URLs from the BadURLManager, and if they are available, removing
them from the BadURLManager and placing them in the DiskCache.

@author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
@version $Id: BadURLManagerDaemon.java,v 1.10 2001/12/17 15:32:24 sgala Exp $
*/
public class BadURLManagerDaemon implements Daemon {
    
    private int status = Daemon.STATUS_NOT_PROCESSED;
    private int result = Daemon.RESULT_UNKNOWN;
    private DaemonConfig config = null;
    private DaemonEntry entry = null;
    
    /**
    Go over all the documents on the system and if refresh them if necessary.
    */
    public void run() {
        
        Log.info("Checking for valid URLs within the URLManager");

        this.setResult( Daemon.RESULT_PROCESSING );
        
        Iterator i = URLManager.list().iterator();
        
        while ( i.hasNext() ) {

            String url = (String)i.next();
            
            // we only want to process bad URLs...
            if ( URLManager.isOK( url ) ) continue;
            
            try {
    
                URLManager.unregister(url);

                Log.info("Removing " + url + " from BadURL list" );
                this.save();

                //FIXME: It should check refresh of the portlets, like DiskCache...
                URLFetcher.refresh(url);

                
            } catch ( Throwable t ) {
                //don't do anything here because the URL for this has a good
                //chance of being invalid anyway.
                Log.error( t );
                if ( DiskCacheUtils.isCached( url ) ) {
                    try {
                        //To avoid interference with the Disk Cache refreshing
                        JetspeedDiskCache.getInstance().remove( url );
                    } catch (IOException ioe) {}
                }
                Log.info("Failed to load: " + url + " from BadURL list");
            }
            
        }
        
        this.setResult( Daemon.RESULT_SUCCESS );
    }

    /**
    */
    public void init( DaemonConfig config, 
                      DaemonEntry entry ) {
        this.config = config;
        this.entry = entry;
    }
    
    /**
    */
    public DaemonConfig getDaemonConfig() {
        return this.config;
    }

    /**
    */
    public DaemonEntry getDaemonEntry() {
        return this.entry;
    }
    
    /**
    Return the status for this Daemon

    @see Daemon#STATUS_NOT_PROCESSED
    @see Daemon#STATUS_PROCESSED
    @see Daemon#STATUS_PROCESSING
    */
    public int getStatus() {
        return this.status;
    }
    
    /**
    Set the status for this Daemon

    @see #STATUS_NOT_PROCESSED
    @see #STATUS_PROCESSED
    @see #STATUS_PROCESSING
    */
    public void setStatus(int status) {
        this.status = status;
    }

    /**
    @see Daemon#getResult()
    */
    public int getResult() {
        return this.result;
    }

    /**
    @see Daemon#setResult(int result)
    */
    public void setResult( int result ) {
        this.result = result;
    }
    
    /**
    @see Daemon#getMessage()
    */
    public String getMessage() {
        return null;
    }

    /**
    Require that the BadURLManager save its configuration here.
    */
    public void save() {

        // RL: What should be persisted here ?
        // BadURLManager.getInstance().save();

    }

    public void restore() { /* noop */ }

    
    
}

