/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.test;

// Java imports
import java.io.File;
import java.io.FileReader;
import java.io.Writer;
import java.io.FileWriter;
import java.io.IOException;

// ECS imports
import org.apache.ecs.ConcreteElement;

// Junit imports
import junit.awtui.TestRunner;
import junit.framework.Test;
import junit.framework.TestSuite;

// Cactus imports
import org.apache.cactus.ServletTestCase;
import org.apache.cactus.WebRequest;
import org.apache.cactus.WebResponse;

// Jetspeed imports
import org.apache.jetspeed.capability.CapabilityMap;
import org.apache.jetspeed.om.profile.Control;
import org.apache.jetspeed.om.profile.Controller;
import org.apache.jetspeed.om.profile.Profile;
import org.apache.jetspeed.om.profile.ProfileLocator;
import org.apache.jetspeed.om.profile.Portlets;
import org.apache.jetspeed.om.profile.PSMLDocument;
import org.apache.jetspeed.om.profile.psml.PsmlController;
import org.apache.jetspeed.om.profile.psml.PsmlEntry;
import org.apache.jetspeed.om.profile.psml.PsmlPortlets;
import org.apache.jetspeed.om.profile.psml.PsmlRole;
import org.apache.jetspeed.om.profile.psml.PsmlSkin;
import org.apache.jetspeed.services.Profiler;
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.rundata.JetspeedRunData;
import org.apache.jetspeed.test.TurbineTestUtilities;

// Turbine imports
import org.apache.turbine.om.security.User;
import org.apache.turbine.services.pull.TurbinePull;
import org.apache.turbine.services.velocity.TurbineVelocity;
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.RunDataFactory;

/**
 * TestBasicSecurity
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id: TestBasicSecurity.java,v 1.3 2002/11/08 10:05:19 raphael Exp $
 */

public class TestBasicSecurity extends ServletTestCase
{
    private static String ADMIN_PORTLET = "GlobalAdminPortlet"; // Portlet accessable by Admin user, role = admin
    private static String ALL_PORTLET = "HelloVelocity";  // Portlet accessable by Anonymous user
    private static String TEST_ADMIN_PASSWORD ="jetspeed";
    private static String TEST_ADMIN_USER = "admin";
    private static String TEST_CONTEXT = null;
    private static String TEST_HOST = "localhost";
    private static String TEST_SERVLET = "/portal";
    private static String TEST_GROUP = "Jetspeed";
    private static String TEST_SECURITY_PAGE = "SecurityTest";
    private static String TEST_TURBINE_PASSWORD ="turbine";
    private static String TEST_TURBINE_USER = "turbine";
    private static String USER_PORTLET = "SkinBrowser"; // Portlet accessable by general user, role = user

    private RunData rundata = null;

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestBasicSecurity(String name)
    {
        super( name );
    }

    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[])
    {
        TestRunner.main( new String[]
        { TestBasicSecurity.class.getName() } );
    }

    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite()
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestBasicSecurity.class );
    }

    /**
     * Sets up the test case.
     *
     */
    protected void setUp() throws Exception
    {
    }

    /**
     *  Test: PageURL
     *  With the page URL "/page/apache"
     *    1) A page is generated
     *    2) The user is anonymous
     *    3) Group is set to "apache"
     *    4) Role is not set
     */
    public void beginCreateTestPSML(WebRequest theRequest)
    {
        System.out.println("URL = " + theRequest.getURL());
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
        , null ,"action=JLoginUser&username="+  TEST_ADMIN_USER + "&password=" + TEST_ADMIN_PASSWORD);
        System.out.println("post set URL = " + theRequest.getURL());
    }

    /**
     * Create a PSML in the TEST_GROUP with 3 entries.
     *  1) ST_01.all  no role
     *  2) ST_01.user  Role = user
     *  3) ST_01.admin  Role = admin
     *
     * @throws Exception
     */
    public void testCreateTestPSML() throws Exception
    {
        Portlets rootPortletSet = null;
        ProfileLocator currentLocator = null;
        ProfileLocator newLocator = null;
        PsmlController controller = null;
        PsmlPortlets portlets = null;
        PsmlSkin skin = null;

        // Create the RunData object to be used during testing.
        rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);

        TurbineTestUtilities.setupRunData(rundata);
        // Verify we have a profile
        Profile profile = Profiler.getProfile(rundata);
        assertNotNull( "Got profile from Profiler", profile);

        // Verify the profile location information in the profile
        if (profile instanceof ProfileLocator)
        {
            currentLocator = (ProfileLocator) profile;
        }

        newLocator = Profiler.createLocator();
        newLocator.setGroupByName(TEST_GROUP);
        newLocator.setMediaType(currentLocator.getMediaType());
        newLocator.setName(TEST_SECURITY_PAGE);

        // Create portlet set
        portlets = new PsmlPortlets();
        controller = new PsmlController();
        controller.setName("RowController");
        portlets.setController(controller);
        skin = new PsmlSkin();
        skin.setName("orange-red");
        portlets.setSkin(skin);
        rootPortletSet = portlets;

        portlets = new PsmlPortlets();
        // Add entries
        portlets.addEntry( createEntry(ALL_PORTLET, "ST_01.all"));
        portlets.addEntry( createEntry(USER_PORTLET, "ST_01.user"));
        portlets.addEntry( createEntry(ADMIN_PORTLET, "ST_01.admin"));
        rootPortletSet.addPortlets(portlets);

        Profile newProfile = Profiler.createProfile(newLocator, rootPortletSet);
        PSMLDocument doc = newProfile.getDocument();
        System.out.println("doc = " + doc.getName());

        // this only works with the default configuration (Castor/Filebased)
        File file = new File(doc.getName());
        assertTrue(file.exists());
        //file.delete();

        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);

        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }

    public void endCreateTestPSML(WebResponse theResponse)
    {
        System.out.println("text length = " + theResponse.getText().length());
        //        System.out.println("text length = " + theResponse.getText());
    }

    /**
     * This test verifies that no "Actions" like minimize are displayed for
     * the anonymous user.  The ID is used to determine if an action is on
     * the page for an entry.
     */
    public void beginAnonymousUser(WebRequest theRequest)
    {
        System.out.println("URL = " + theRequest.getURL());
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
        , "/group/" + TEST_GROUP + "/page/" + TEST_SECURITY_PAGE , null );
        System.out.println("post set URL = " + theRequest.getURL());
    }

    public void testAnonymousUser() throws Exception
    {
        // Create the RunData object to be used during testing.
        rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);

        TurbineTestUtilities.setupRunData(rundata);

        // Verify we have a profile
        Profile profile = Profiler.getProfile(rundata);
        assertNotNull( "Got profile from Profiler", profile);

        // Verify the profile location information in the profile
        if (profile instanceof ProfileLocator)
        {
            ProfileLocator profileLocator = (ProfileLocator) profile;
            assertTrue("Verify the 'anonymous' is not set", !profileLocator.getAnonymous());
            assertNull("Verify the user is null", profileLocator.getUser());
            assertNull("Verify the role is null", profileLocator.getRole());
            assertEquals("Verify the page name", profileLocator.getName(), TEST_SECURITY_PAGE + ".psml");
        } else
        {
            assertTrue( "profile does not implement ProfileLocator", false);
        }
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);

        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }

    public void endAnonymousUser(WebResponse theResponse)
    {
        assertEquals("Verify resulting page does not contain 'ST_01.all'", theResponse.getText().indexOf("ST_01.all"),-1);
        assertEquals("Verify resulting page does not contain 'ST_01.user'", theResponse.getText().indexOf("ST_01.user"),-1);
        assertEquals("Verify resulting page does not contain 'ST_01.admin'", theResponse.getText().indexOf("ST_01.admin"), -1);
        System.out.println("text length = " + theResponse.getText().length());
        //        System.out.println("text length = " + theResponse.getText());
    }

    /**
     * This test verifies that only protlets with a role of User has "Actions"
     * like minimize are displayed a logged in user with a role = user.  The
     * ID is used to determine if an action is on the page for an entry.
     */
    public void beginTurbineUser(WebRequest theRequest)
    {
        System.out.println("URL = " + theRequest.getURL());
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
        , "/group/" + TEST_GROUP + "/page/" + TEST_SECURITY_PAGE , "action=JLoginUser&username="+  TEST_TURBINE_USER + "&password=" + TEST_TURBINE_PASSWORD);
        System.out.println("post set URL = " + theRequest.getURL());
    }

    public void testTurbineUser() throws Exception
    {
        // Create the RunData object to be used during testing.
        rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);

        TurbineTestUtilities.setupRunData(rundata);

        // Verify we have a profile
        Profile profile = Profiler.getProfile(rundata);
        assertNotNull( "Got profile from Profiler", profile);

        // Verify the profile location information in the profile
        if (profile instanceof ProfileLocator)
        {
            ProfileLocator profileLocator = (ProfileLocator) profile;
            assertTrue("Verify the 'anonymous' is not set", !profileLocator.getAnonymous());
            assertNull("Verify the user is null", profileLocator.getUser());
            assertNull("Verify the role is null", profileLocator.getRole());
            assertEquals("Verify the page name", profileLocator.getName(), TEST_SECURITY_PAGE + ".psml");
        } else
        {
            assertTrue( "profile does not implement ProfileLocator", false);
        }
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);

        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }

    public void endTurbineUser(WebResponse theResponse)
    {
        assertTrue("Verify resulting page contains 'ST_01.all'", theResponse.getText().indexOf("ST_01.all") > -1);
        assertTrue("Verify resulting page contains 'ST_01.user'", theResponse.getText().indexOf("ST_01.user") > -1);
        assertEquals("Verify resulting page does not contain 'ST_01.admin'", theResponse.getText().indexOf("ST_01.admin"), -1);
        System.out.println("text length = " + theResponse.getText().length());
        //        System.out.println("text length = " + theResponse.getText());
    }

    /**
     * This test verifies that only protlets with a role of User or admin have
     * "Actions" like minimize are displayed a logged in user with a
     * role = user and admin.  The ID is used to determine if an action is on
     * the page for an entry.
     */
    public void beginAdminUser(WebRequest theRequest)
    {
        System.out.println("URL = " + theRequest.getURL());
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
        , "/group/" + TEST_GROUP + "/page/" + TEST_SECURITY_PAGE , "action=JLoginUser&username="+  TEST_ADMIN_USER + "&password=" + TEST_ADMIN_PASSWORD);
        System.out.println("post set URL = " + theRequest.getURL());
    }

    public void testAdminUser() throws Exception
    {
        // Create the RunData object to be used during testing.
        rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);

        TurbineTestUtilities.setupRunData(rundata);

        // Verify we have a profile
        Profile profile = Profiler.getProfile(rundata);
        assertNotNull( "Got profile from Profiler", profile);

        // Verify the profile location information in the profile
        if (profile instanceof ProfileLocator)
        {
            ProfileLocator profileLocator = (ProfileLocator) profile;
            assertTrue("Verify the 'anonymous' is not set", !profileLocator.getAnonymous());
            assertNull("Verify the user is null", profileLocator.getUser());
            assertNull("Verify the role is null", profileLocator.getRole());
            assertEquals("Verify the page name", profileLocator.getName(), TEST_SECURITY_PAGE + ".psml");
        } else
        {
            assertTrue( "profile does not implement ProfileLocator", false);
        }
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);

        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }

    public void endAdminUser(WebResponse theResponse)
    {
        assertTrue("Verify resulting page contains 'ST_01.all'", theResponse.getText().indexOf("ST_01.all") > -1);
        assertTrue("Verify resulting page contains 'ST_01.user'", theResponse.getText().indexOf("ST_01.user") > -1);
        assertTrue("Verify resulting page contains 'ST_01.admin'", theResponse.getText().indexOf("ST_01.admin")> -1);
        System.out.println("text length = " + theResponse.getText().length());
        //        System.out.println("text length = " + theResponse.getText());
    }

    private PsmlEntry createEntry(java.lang.String parent, java.lang.String id)
    {
        PsmlEntry entry = new PsmlEntry();
        entry.setParent( parent);
        if (id != null)
            entry.setId( id);
        return entry;
    }

}
