/**********************************************************************************
*
* $Header: /home/cvspublic/jakarta-jetspeed/src/java/org/apache/jetspeed/services/statemanager/StateManagerService.java,v 1.4 2002/08/08 00:05:18 ggolden Exp $
*
* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// package
package org.apache.jetspeed.services.statemanager;

// imports
import javax.servlet.http.HttpSession;

/**
* <p>The StateManagerService is a service that manages SessionState information.
* Each SessionState is identified by a unique key in this service.  The SessionState
* is composed of name - value sets of attributes, stored under the key by the service.</p>
* <p>See the proposal: jakarta-jetspeed/proposals/StateManager.txt for more details.</p>
* <p>Attribute values placed into SessionStates may be automatically removed, for example when
* a SessionState expires in some way, or by means of some other automatic service.  These value
* objects can be notified of their placement into and out of the SessionState.  Objects that wish
* to receive this notification implement the SessionStateBindingListener interface.  This is based
* on and similar to the HttpSessionBindingListener mechanism.</p>
* <p>To support the "current" routines; the ability to get a session state based
* on the current http session, this service must be installed properly into the
* procssing of each http request.  At the start of the request, the HttpSession
* of the request must be passed into the setCurrentContext() method.  At the end
* of the request, clearCurrentContext() is called.  For Jetspeed, this is done in
* the JetspeedRunDataService, which is always going to be called by Turbine in this
* way.</p>
* @version $Revision: 1.4 $
* @see org.apache.jetspeed.services.statemanager.SessionState
* @see org.apache.jetspeed.services.statemanager.SessionStateBindingListener
* @author <a href="mailto:ggolden@apache.org">Glenn R. Golden</a>
*/
public interface StateManagerService
{
    /** The name used to find the service in the service manager. */
    public String SERVICE_NAME = "StateManagerService";

    /**
    * Access the named attribute of the keyed state.
    * @param key The state key.
    * @param name The attribute name.
    * @return The named attribute value of the keyed state.
    */
    public Object getAttribute ( String key, String name );

    /**
    * Set the named state attribute of the keyed state with the provided object.
    * @param key The state key.
    * @param name The attribute name.
    * @param value The new value of the attribute (any object type).
    */
    public void setAttribute( String key, String name, Object value );

    /**
    * Remove the named state attribute of the keyed state, if it exists.
    * @param key The state key.
    * @param name The attribute name.
    */
    public void removeAttribute( String key, String name );

    /**
    * Remove all state attribute of the keyed state.
    * @param key The state key.
    */
    public void clear( String key );

    /**
    * Access an array of all names of attributes stored in the keyed state.
    * @param key The state key.
    * @return An array of all names of attributes stored in the keyed state.
    */
    public String[] getAttributeNames( String key );

    /**
    * Access an SessionState object with the given key.
    * @param key The SessionState key.
    * @return an SessionState object with the given key.
    */
    public SessionState getSessionState( String key );

    /**
    * Access the SessionState object associated with the current request's http session.
	* The session id is used as the key.
    * @return an SessionState object associated with the current request's http session.
    */
    public SessionState getCurrentSessionState();

    /**
    * Access the SessionState object associated with the current request's http session with the given key.
	* @param key The string to add to the session id to form the SessionState key.
    * @return an SessionState object associated with the current request's http session with the given key.
    */
    public SessionState getCurrentSessionState( String key );

    /**
    * Retire, forget about and clean up all states that start with the given key.
    * @param keyStart The beginning of the key of the states to clean up.
    */
    public void retireState( String keyStart );

    /**
    * Set the "current" context for this thread -
    * Call this at the start of each request, and call %%% at the end.
    * getCurrentSession() uses this for the session state key.
    * @param session the HttpSession of the current request.
    */
    public void setCurrentContext( HttpSession session );

    /**
    * Clear the "current context for this thread -
    * Call at the end of each request, balanced with calls to setCurrentContext()
    */
    public void clearCurrentContext();

}   // interface StateManagerService

/**********************************************************************************
*
* $Header: /home/cvspublic/jakarta-jetspeed/src/java/org/apache/jetspeed/services/statemanager/StateManagerService.java,v 1.4 2002/08/08 00:05:18 ggolden Exp $
*
**********************************************************************************/

