/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security.registry;

// Java imports
import java.util.Iterator;
import javax.servlet.ServletConfig;

// Jetspeed import
import org.apache.jetspeed.om.SecurityReference;
import org.apache.jetspeed.om.profile.Entry;
import org.apache.jetspeed.om.registry.RegistryEntry;
import org.apache.jetspeed.om.registry.SecurityEntry;
import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.portal.Portlet;
import org.apache.jetspeed.portal.PortletController;
import org.apache.jetspeed.portal.PortletSet;
import org.apache.jetspeed.services.resources.JetspeedResources;
import org.apache.jetspeed.services.Registry;
import org.apache.jetspeed.services.security.JetspeedRoleManagement;
import org.apache.jetspeed.services.security.PortalAccessController;
import org.apache.jetspeed.services.security.PortalResource;

// Turbine imports
import org.apache.turbine.services.TurbineBaseService;
import org.apache.turbine.services.InitializationException;
import org.apache.turbine.util.Log;

/**
 * TurbineAccessController
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id: RegistryAccessController.java,v 1.4 2002/08/25 19:45:07 paulsp Exp $
 */
public class RegistryAccessController extends TurbineBaseService implements PortalAccessController
{
    
    /**
     * Given a <code>JetspeedUser</code>, authorize that user to perform the secured action on
     * the given <code>Portlet</code> resource. If the user does not have
     * sufficient privilege to perform the action on the resource, the check returns false,
     * otherwise when sufficient privilege is present, checkPermission returns true.
     *
     * @param user the user to be checked.
     * @param portlet the portlet resource.
     * @param action the secured action to be performed on the resource by the user.
     * @return boolean true if the user has sufficient privilege.
     */
    public boolean checkPermission(JetspeedUser user, Portlet portlet, String action)
    {
        return checkPermission(user, portlet, action, null);
    }

    /**
     * Given a <code>JetspeedUser</code>, authorize that user to perform the secured action on
     * the given <code>Portlet</code> resource. If the user does not have
     * sufficient privilege to perform the action on the resource, the check returns false,
     * otherwise when sufficient privilege is present, checkPermission returns true.
     *
     * @param user the user to be checked.
     * @param portlet the portlet resource.
     * @param action the secured action to be performed on the resource by the user.
     * @param owner of the entry, i.e. the username
     * @return boolean true if the user has sufficient privilege.
     */
    public boolean checkPermission(JetspeedUser user, Portlet portlet, String action, String owner)
    {
        SecurityReference securityRef = portlet.getPortletConfig().getSecurityRef();
        if (securityRef != null)
        {
            return checkPermission( user, securityRef, action, owner);
        }

        String portletName = portlet.getName();
        RegistryEntry registryEntry = (RegistryEntry)Registry.getEntry(Registry.PORTLET, portletName);
        //portlet is not a portlet - probably a controller or control
        if (registryEntry==null) {
            PortletSet ps  = portlet.getPortletConfig().getPortletSet();
            if (ps != null) {
                PortletController pc = ps.getController();
                if (pc != null) {
                    portletName = pc.getConfig().getName();
                    registryEntry = (RegistryEntry)Registry.getEntry(Registry.PORTLET_CONTROLLER, portletName);
                }
            }
        }
        if (registryEntry==null) {
            return true; // Since their is no entry, their no security to test.  Per spec. all is allowed
        }

        return checkPermission(user, registryEntry, action, owner);
    }
    
    /**
     * Given a <code>JetspeedUser</code>, authorize that user to perform the secured action on
     * the given Portlet Instance (<code>Entry</code>) resource. If the user does not have
     * sufficient privilege to perform the action on the resource, the check returns false,
     * otherwise when sufficient privilege is present, checkPermission returns true.
     *
     * @param user the user to be checked.
     * @param entry the portlet instance resource.
     * @param action the secured action to be performed on the resource by the user.
     * @return boolean true if the user has sufficient privilege.
     */
    public boolean checkPermission(JetspeedUser user, Entry entry, String action)
    {
        return checkPermission( user, entry, action, null);
    }

    /**
     * Given a <code>JetspeedUser</code>, authorize that user to perform the secured action on
     * the given Portlet Instance (<code>Entry</code>) resource. If the user does not have
     * sufficient privilege to perform the action on the resource, the check returns false,
     * otherwise when sufficient privilege is present, checkPermission returns true.
     *
     * @param user the user to be checked.
     * @param entry the portlet instance resource.
     * @param action the secured action to be performed on the resource by the user.
     * @param owner of the entry, i.e. the username
     * @return boolean true if the user has sufficient privilege.
     */
    public boolean checkPermission(JetspeedUser user, Entry entry, String action, String owner)
    {
        SecurityReference securityRef = entry.getSecurityRef();
        if (securityRef == null)
        {
            return checkPermission( user, Registry.getEntry( Registry.PORTLET, entry.getParent()), action, owner);
        }
        return checkPermission( user, securityRef, action, owner);
    }
       
    
    /**
     * Given a <code>JetspeedUser</code>, authorize that user to perform the secured action on
     * the given resource. If the user does not have
     * sufficient privilege to perform the action on the resource, the check returns false,
     * otherwise when sufficient privilege is present, checkPermission returns true.
     *
     * @param user the user to be checked.
     * @param resource requesting an action
     * @param action the secured action to be performed on the resource by the user.
     * @return boolean true if the user has sufficient privilege.
     */
    public boolean checkPermission(JetspeedUser user, PortalResource resource, String action)
    {
        switch (resource.getResourceType())
        {
            case PortalResource.TYPE_ENTRY:
                return checkPermission(user, resource.getEntry(), action, resource.getOwner());
            case PortalResource.TYPE_PORTLET:
                return checkPermission(user, resource.getPortlet(), action, resource.getOwner());
            case PortalResource.TYPE_REGISTRY:
                return checkPermission(user, resource.getRegistryEntry(), action, resource.getOwner());
            case PortalResource.TYPE_REGISTRY_PARAMETER:
                return checkPermission(user, resource.getRegistryParameter(), action, resource.getOwner());
        }

        // We should never get here
        Log.error( "In " + this.getClass().getName() + ".checkPermission(user, resource, action) - Unkown resource = " + resource.getResourceType());
        return false;
    }
    
    /**
     * Checks if the user has access to a given registry entry for the given action
     *
     * @param user the requesting user.
     * @param regEntry the registry entry from the registry.
     * @param owner of the entry, i.e. the username
     * @param action the jetspeed-action (view, edit, customize, delete...) for which permission is being checked.
     */
    private boolean checkPermission(JetspeedUser user,  RegistryEntry regEntry, String action, String owner)
    {
        SecurityReference securityRef = regEntry.getSecurityRef();
        if (securityRef == null)
            return true;  // No security defined on Registry entry
        return checkPermission( user, securityRef, action, owner);
    }

    /**
     * Checks if the user has access for the given action using a security reference 
     *
     * @param user the requesting user.
     * @param securityRef the security reference to check
     * @param action the jetspeed-action (view, edit, customize, delete...) for which permission is being checked.
     */
    private boolean checkPermission(JetspeedUser user, SecurityReference securityRef, String action, String owner)
    {
        SecurityEntry securityEntry = (SecurityEntry) Registry.getEntry( Registry.SECURITY, securityRef.getParent());
        if (securityEntry == null)
        {
            Log.warn("Security id " + securityRef.getParent() + " does not exist.  This was requested by the user " + user.getUserName());
            return false;
        }

        if (securityEntry.allowsUser(user.getUserName(), action, owner))
        {
            return true;
        }

        try
        {
            for( Iterator roles = JetspeedRoleManagement.getRoles(user.getUserName()); roles.hasNext();)
            {
                Role role = (Role) roles.next();
                if (securityEntry.allowsRole((String) role.getName(), action))
                    return true;
            }
        }
        catch (Exception e)
        {
            Log.error(e);
            return false;
        }
        return false;
    }
    
    /*
     * Turbine Services Interface
     */
    
    /**
     * This is the early initialization method called by the
     * Turbine <code>Service</code> framework
     * @param conf The <code>ServletConfig</code>
     * @exception throws a <code>InitializationException</code> if the service
     * fails to initialize
     */
    public synchronized void init(ServletConfig conf)
    throws InitializationException
    {
        if (getInit()) return;
        
        super.init(conf);
        
        setInit(true);
    }
}
