/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security.ldap;

import java.util.Iterator;
import java.util.Vector;
import java.util.HashMap;
import java.util.Enumeration;
import javax.servlet.ServletConfig;


// Jetspeed Security
import org.apache.jetspeed.services.security.PermissionManagement;
import org.apache.jetspeed.services.security.JetspeedSecurityCache;
import org.apache.jetspeed.services.security.CachedAcl;

import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.om.security.BaseJetspeedUser;
import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.om.security.Permission;

import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.security.UserManagement;
import org.apache.jetspeed.services.security.JetspeedSecurityService;
import org.apache.jetspeed.om.security.UserNamePrincipal;
import org.apache.jetspeed.om.security.BaseJetspeedPermission;

// Jetspeed Security Exceptions
import org.apache.jetspeed.services.security.PermissionException;
import org.apache.jetspeed.services.security.JetspeedSecurityException;

import org.apache.jetspeed.services.PsmlManager;

//openisp
import javax.naming.*;
import javax.naming.directory.*;

import org.apache.jetspeed.om.security.ldap.LDAPPermission;
import org.apache.jetspeed.om.security.ldap.LDAPUser;
import org.apache.jetspeed.om.security.ldap.LDAPRole;
import org.apache.jetspeed.services.JetspeedLDAP;
import org.apache.jetspeed.services.ldap.LDAPURL;


// Rundata
import org.apache.jetspeed.services.rundata.JetspeedRunDataService;
import org.apache.jetspeed.services.rundata.JetspeedRunData;
import org.apache.turbine.services.rundata.RunDataService;

// Turbine
import org.apache.turbine.util.Log;
import org.apache.turbine.services.TurbineBaseService;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.services.InitializationException;
import org.apache.turbine.services.resources.ResourceService;

/**
 *
 * @author <a href="mailto:ender@kilicoglu.nom.tr">Ender KILICOGLU</a>
 * @version $Id: LDAPPermissionManagement.java,v 1.3 2002/12/09 20:24:08 taylor Exp $ 
 * 
 */
public class LDAPPermissionManagement  extends TurbineBaseService
                                   implements PermissionManagement
{
    private JetspeedRunDataService runDataService = null;
    private final static String CASCADE_DELETE = "programmatic.cascade.delete";
    private final static boolean DEFAULT_CASCADE_DELETE = true;
    private boolean cascadeDelete;
    private final static String CACHING_ENABLE = "caching.enable";
    private boolean cachingEnable = true;
    String[] attrs = {"ou", "uid"};

    ///////////////////////////////////////////////////////////////////////////
    // Permission Management Interfaces
    ///////////////////////////////////////////////////////////////////////////

    /**
     * Retrieves all <code>Permission</code>s for a given roleName principal.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param roleName a role name identity to be retrieved.
     * @return Iterator over all permissions associated to the role principal.
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public Iterator getPermissions(String roleName)
        throws JetspeedSecurityException
    {
        Vector perms = new Vector();
        BasicAttributes attr= new BasicAttributes();
        LDAPRole role;
        LDAPPermission permission;
        Vector userurls;

        try
        {
            if (cachingEnable)
            {
                Iterator iterator = JetspeedSecurityCache.getPermissions(roleName);
                if (iterator != null)
                {
                    return iterator;
                }
            }

            userurls = JetspeedLDAP.search(JetspeedLDAP.buildURL("ou=roles"),
                       "(&(uid="+ roleName+")(objectclass=jetspeedrole))",attrs,true);

            if (userurls.size() > 0)
            {
                role = new LDAPRole((LDAPURL) ((Vector)userurls.elementAt(0)).firstElement());
                for (Enumeration enum= role.getRolePermissions().elements();enum.hasMoreElements();)
                {
                    permission = new LDAPPermission((String)enum.nextElement(), false);
                    perms.add(permission);
                }
            }
        }
        catch(Exception e)
        {
            throw new PermissionException("Failed to retrieve permissions ", e);
        }

        return perms.iterator();
    }

    /**
     * Retrieves all <code>Permission</code>s.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @return Iterator over all permissions.
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public Iterator getPermissions()
        throws JetspeedSecurityException








    {
        BasicAttributes attr= new BasicAttributes();
        Vector permissions = new Vector();
        Vector permissionurls;
 
        try
        {
            permissionurls = JetspeedLDAP.search(JetspeedLDAP.buildURL("ou=permissions"),
                             "(objectclass=jetspeedpermission)",attrs,true);

            if (permissionurls.size() > 0)
            {
                for (Enumeration enum = permissionurls.elements();enum.hasMoreElements() ;)
                {
                    permissions.add(new LDAPPermission((LDAPURL) (((Vector)enum.nextElement()).firstElement())));
                }
            }
            else
            {
                throw new PermissionException("No permission ");
            }
        }
        catch(Exception e)
        {
            throw new PermissionException("Failed to retrieve permissions ", e);
        }
        return permissions.iterator();
    }

    /**
     * Adds a <code>Permission</code> into permanent storage.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void addPermission(Permission permission)
        throws JetspeedSecurityException
    {
        if(permissionExists(permission.getName()))
        {
            throw new PermissionException("The permission '" +
                permission.getName() + "' already exists");
        }
        try
        {
          new LDAPPermission(permission.getName(),true).update(true) ;
        }
        catch(Exception e)
        {
            throw new PermissionException("Failed to create permission '" +
                permission.getName() + "'", e);
        }
    }


    /**
     * Saves a <code>Permission</code> into permanent storage.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void savePermission(Permission permission)
        throws JetspeedSecurityException
    {
    }

    /**
     * Removes a <code>Permission</code> from the permanent store.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param permissionName the principal identity of the permission to be retrieved.
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void removePermission(String permissionName)
        throws JetspeedSecurityException
    {
        try
        {
            LDAPPermission permission = new LDAPPermission(permissionName,false);
            JetspeedLDAP.deleteEntry(permission.getldapurl());

            if(cascadeDelete)
            {
            }

            if (cachingEnable)
            {
                JetspeedSecurityCache.removeAllPermissions(permissionName);
            }
        }
        catch(Exception e)
        {
            throw new PermissionException("Failed to remove permission '" +
                permissionName + "'", e);
        }
    }

     /**
     * Grants a permission to a role.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param roleName grant a permission to this role.

     * @param permissionName the permission to grant to the role.
     * @exception PermissionException when the security provider has a general failure retrieving permissions.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void grantPermission(String roleName, String permissionName)
        throws JetspeedSecurityException
    {
        BasicAttributes attr = new BasicAttributes();
        LDAPRole role;
        LDAPPermission permission;

        try
        {
            role = (LDAPRole)JetspeedSecurity.getRole(roleName);
            permission = (LDAPPermission)JetspeedSecurity.getPermission(permissionName);

            role.addRolePermissions(permissionName);
            role.update(false);

            if (cachingEnable)
            {
                JetspeedSecurityCache.addPermission(roleName, permission);
            }

        }
        catch(Exception e)
        {
            throw new PermissionException("Grant permission '" + permissionName + "' to role '" + roleName + "' failed: ", e);
        }
    }

    /**
     * Revokes a permission from a role.
     *

     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param roleName grant a permission to this role.
     * @param permissionName the permission to grant to the role.
     * @exception PermissionException when the security provider has a general failure retrieving permissions.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void revokePermission(String roleName, String permissionName)
        throws JetspeedSecurityException
    {
        BasicAttributes attr= new BasicAttributes();
        LDAPRole role;
        Vector userurls;

        try
        {
            userurls = JetspeedLDAP.search(JetspeedLDAP.buildURL("ou=roles"),
                       "(&(uid="+ roleName+")(objectclass=jetspeedrole))",attrs,true);
            if (userurls.size() > 0)
            {
                role = new LDAPRole((LDAPURL) ((Vector)userurls.elementAt(0)).firstElement());
                role.getRolePermissions().remove(permissionName);
                role.update(false);

                if (cachingEnable)
                {
                    JetspeedSecurityCache.removePermission(roleName, permissionName);
                }
            }
        }
        catch(Exception e)
        {
            throw new PermissionException("Revoke permission '" + permissionName + "' to role '" + roleName + "' failed: ", e);
       }
    }

    /**
     * Checks for the relationship of role has a permission. Returns true when the role has the given permission.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param roleName grant a permission to this role.
     * @param permissionName the permission to grant to the role.
     * @exception PermissionException when the security provider has a general failure retrieving permissions.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public boolean hasPermission(String roleName, String permissionName)
        throws JetspeedSecurityException
    {
        BasicAttributes attr= new BasicAttributes();
        LDAPRole role;
        Vector userurls;

        try
        {
            if (cachingEnable)
            {
                return JetspeedSecurityCache.hasPermission(roleName, permissionName);
            }

            userurls = JetspeedLDAP.search(JetspeedLDAP.buildURL("ou=roles"),
                       "(&(uid="+ roleName+")(objectclass=jetspeedrole))",attrs,true);
            if (userurls.size() > 0)
            {
                role = new LDAPRole((LDAPURL) ((Vector)userurls.elementAt(0)).firstElement());

                return role.permissionExists(permissionName);
            }
        }
        catch(Exception e)
        {
            throw new PermissionException("Grant permission '" + permissionName + "' to role '" + roleName + "' failed: ", e);
        }
        return false;
    }

    /**
     * Retrieves a single <code>Permission</code> for a given permissionName principal.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param permissionName a permission principal identity to be retrieved.
     * @return Permission the permission record retrieved.
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public Permission getPermission(String permissionName)
        throws JetspeedSecurityException
    {
        if (permissionExists(permissionName))
        {
            return new LDAPPermission(permissionName, false);
        }
        else
        {
            throw new PermissionException("Unknown permission '" + permissionName + "'");
        }
    }

    ///////////////////////////////////////////////////////////////////////////
    // Internal
    ///////////////////////////////////////////////////////////////////////////

    protected JetspeedRunData getRunData()
     {
         JetspeedRunData rundata = null;
         if (this.runDataService != null)
         {
             rundata = this.runDataService.getCurrentRunData();
         }
         return rundata;
     }

    /**
     * Check whether a specified permission exists.
     *
     * The login name is used for looking up the account.
     *
     * @param permissionName the name of the permission to check for existence.
     * @return true if the specified account exists
     * @throws PermissionException if there was a general db access error
     *
     */
    protected boolean permissionExists(String permissionName)
        throws PermissionException
    {
        BasicAttributes attr= new BasicAttributes();
        Vector permissionurls;

        try
        {
            permissionurls = JetspeedLDAP.search(JetspeedLDAP.buildURL("ou=permissions"),
                             "(&(uid=" + permissionName + ")(objectclass=jetspeedpermission))",attrs,true);

            if (permissionurls.size() > 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        }
        catch(Exception e)
        {
            e.printStackTrace();
            throw new PermissionException("Failed to retrieve permission ", e);
        }
    }

    ///////////////////////////////////////////////////////////////////////////
    // Service Init
    ///////////////////////////////////////////////////////////////////////////

    /**
     * This is the early initialization method called by the
     * Turbine <code>Service</code> framework
     * @param conf The <code>ServletConfig</code>
     * @exception throws a <code>InitializationException</code> if the service
     * fails to initialize
     */
    public synchronized void init(ServletConfig conf)
        throws InitializationException
    {
        if (getInit()) return;

        super.init(conf);

        // get configuration parameters from Jetspeed Resources
        ResourceService serviceConf = ((TurbineServices)TurbineServices.getInstance())
                                                     .getResources(JetspeedSecurityService.SERVICE_NAME);

        this.runDataService =
           (JetspeedRunDataService)TurbineServices.getInstance()
               .getService(RunDataService.SERVICE_NAME);

        cascadeDelete = serviceConf.getBoolean( CASCADE_DELETE, DEFAULT_CASCADE_DELETE );
        cachingEnable = serviceConf.getBoolean( CACHING_ENABLE, cachingEnable );
        setInit(true);
     }
}
