/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security.ldap;

import java.util.Iterator;
import java.util.Vector;
import java.util.HashMap;
import java.util.Enumeration;
import java.util.StringTokenizer;

import javax.servlet.ServletConfig;
//openisp
import javax.naming.*;
import javax.naming.directory.*;

import org.apache.jetspeed.om.security.ldap.LDAPGroup;
import org.apache.jetspeed.om.security.ldap.LDAPUser;
import org.apache.jetspeed.services.JetspeedLDAP;
import org.apache.jetspeed.services.ldap.LDAPURL;

// Jetspeed Security
import org.apache.jetspeed.services.security.GroupManagement;

import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.om.security.BaseJetspeedUser;
import org.apache.jetspeed.om.security.BaseJetspeedGroup;
import org.apache.jetspeed.om.security.Group;
import org.apache.jetspeed.om.security.Role;

import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.security.UserManagement;
import org.apache.jetspeed.services.security.JetspeedSecurityService;
import org.apache.jetspeed.om.security.UserNamePrincipal;

// Jetspeed Security Exceptions
import org.apache.jetspeed.services.security.GroupException;
import org.apache.jetspeed.services.security.UnknownUserException;
import org.apache.jetspeed.services.security.NotUniqueUserException;
import org.apache.jetspeed.services.security.JetspeedSecurityException;

import org.apache.jetspeed.services.PsmlManager;

// Profile
import org.apache.jetspeed.om.profile.Profile;
import org.apache.jetspeed.services.Profiler;
import org.apache.jetspeed.om.profile.ProfileException;

// Rundata
import org.apache.jetspeed.services.rundata.JetspeedRunDataService;
import org.apache.jetspeed.services.rundata.JetspeedRunData;
import org.apache.turbine.services.rundata.RunDataService;

// Turbine
import org.apache.turbine.util.Log;
import org.apache.turbine.services.TurbineBaseService;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.services.InitializationException;
import org.apache.turbine.services.resources.ResourceService;

/**
 *
 * @author <a href="mailto:ender@kilicoglu.nom.tr">Ender KILICOGLU</a>
 * @version $Id: LDAPGroupManagement.java,v 1.4 2002/12/09 20:24:08 taylor Exp $
 *
 */
public class LDAPGroupManagement extends TurbineBaseService
                                   implements GroupManagement
{
    private JetspeedRunDataService runDataService = null;

    private final static String CONFIG_DEFAULT_ROLE = "role.default";
    String defaultRole = "user";
    private final static String CASCADE_DELETE = "programmatic.cascade.delete";
    private final static boolean DEFAULT_CASCADE_DELETE = true;
    private boolean cascadeDelete;
    String[] attrs = {"ou","uid"};

    ///////////////////////////////////////////////////////////////////////////
    // Group Management Interfaces
    ///////////////////////////////////////////////////////////////////////////

    /**
     * Retrieves all <code>Group</code>s for a given username principal.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param username a user principal identity to be retrieved.
     * @return Iterator over all groups associated to the user principal.
     * @exception GroupException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public Iterator getGroups(String username)
        throws JetspeedSecurityException
    {
        StringTokenizer st;
        LDAPUser user;

        try
        {
            user = (LDAPUser)JetspeedSecurity.getUser(new UserNamePrincipal(username));
        }
        catch(JetspeedSecurityException e)
        {
            throw new GroupException("Failed to Retrieve User: ", e);
        }
        Vector groups= new Vector();
        try
        {
            for (Enumeration enum = user.getGroupRoles().elements() ;enum.hasMoreElements() ;)
            {
                st = new StringTokenizer((String)enum.nextElement(),",");
                groups.add(new LDAPGroup(st.nextToken(),false));
            }
        }
        catch(Exception e)
        {
            throw new GroupException("Failed to retrieve groups ", e);
        }
        return groups.iterator();
    }
    /**
     * Retrieves all <code>Group</code>s.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @return Iterator over all groups.
     * @exception GroupException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public Iterator getGroups()
        throws JetspeedSecurityException
    {
        BasicAttributes attr= new BasicAttributes();
        Vector groups = new Vector();
        Vector userurls;
        try
        {
            userurls = JetspeedLDAP.search(JetspeedLDAP.buildURL("ou=groups"),
                                        "(objectclass=jetspeedgroup)",attrs,true);
          if (userurls.size() > 0){
            for (Enumeration enum = userurls.elements();enum.hasMoreElements() ;){
              groups.add(new LDAPGroup((LDAPURL) (((Vector)enum.nextElement()).firstElement())));
            }
          } else {
            throw new UnknownUserException("No groups ");
          }
        }
        catch(Exception e)
        {
            throw new GroupException("Failed to retrieve groups ", e);
        }
        return groups.iterator();
    }

    /**
     * Adds a <code>Group</code> into permanent storage.
     *
     * The security service can throw a <code>NotUniqueEntityException</code> when the public
     * credentials fail to meet the security provider-specific unique constraints.
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception GroupException when the security provider has a general failure.
     * @exception NotUniqueEntityException when the public credentials fail to meet
     *                                   the security provider-specific unique constraints.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void addGroup(Group group)
        throws JetspeedSecurityException
    {
        if(groupExists(group.getName()))
        {
            throw new GroupException("The group '" +
                group.getName() + "' already exists");
        }
        try
        {
          new LDAPGroup(group.getName(),true).update(true);
        }
        catch(Exception e)
        {
            throw new GroupException("Failed to create group '" +
                group.getName() + "'", e);
        }

        try
        {
            addDefaultGroupPSML(group);
        }
        catch (Exception e)
        {
            try
            {
                removeGroup(group.getName());
            }
            catch (Exception e2)
            {
            }
            throw new GroupException("failed to add default PSML for Group resource", e);
        }

    }
    protected void addDefaultGroupPSML(Group group)
        throws GroupException
    {
        try
        {
            JetspeedRunDataService runDataService =
               (JetspeedRunDataService)TurbineServices.getInstance()
                   .getService(RunDataService.SERVICE_NAME);
            JetspeedRunData rundata = runDataService.getCurrentRunData();
            Profile profile = Profiler.createProfile();
            profile.setGroup(group);
            profile.setMediaType("html");
            Profiler.createProfile(rundata, profile);
        }
        catch (ProfileException e)
        {
            try
            {
                removeGroup(group.getName());
            }
            catch(Exception e2)
            {
            }
            throw new GroupException("Failed to create Group PSML", e);
        }
    }
    /**
     * Saves a <code>Group</code> into permanent storage.
     *
     * The security service can throw a <code>NotUniqueEntityException</code> when the public
     * credentials fail to meet the security provider-specific unique constraints.
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception GroupException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void saveGroup(Group group)
        throws JetspeedSecurityException
    {
        if(!groupExists(group.getName()))
        {
            throw new GroupException("The group '" +
                group.getName() + "' doesn't exists");
        }

        try
        {
         }
        catch(Exception e)
        {
            throw new GroupException("Failed to create group '" +
                group.getName() + "'", e);
        }

    }

    /**
     * Removes a <code>Group</code> from the permanent store.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param groupname the principal identity of the group to be retrieved.
     * @exception GroupException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void removeGroup(String groupname)
        throws JetspeedSecurityException
    {
        try
        {
            LDAPGroup group = new LDAPGroup(groupname, false);
            JetspeedLDAP.deleteEntry(group.getldapurl());
            if(cascadeDelete)
            {
            }
            PsmlManager.removeGroupDocuments(group);
        }
        catch(Exception e)
        {
            throw new GroupException("Failed to remove group '" +
                groupname + "'", e);
        }

    }

    /**
     * Join a user to a group.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception GroupException when the security provider has a general failure retrieving users.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void joinGroup(String username, String groupname)
        throws JetspeedSecurityException
    {
        LDAPUser user;
        try
        {
            user = (LDAPUser)JetspeedSecurity.getUser(new UserNamePrincipal(username));
        }
        catch(JetspeedSecurityException e)
        {
            throw new GroupException("Failed to Retrieve User: ", e);
        }
        try
        {
          user.addGroupRole(groupname,defaultRole);
          user.update(false);
        }
        catch(Exception e)
        {
            throw new GroupException("Failed to add group info ", e);
        }
    }

    /**
     * Unjoin a user from a group.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception GroupException when the security provider has a general failure retrieving users.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public void unjoinGroup(String username, String groupName)
        throws JetspeedSecurityException
    {
        LDAPUser user;
        try
        {
            user = (LDAPUser)JetspeedSecurity.getUser(new UserNamePrincipal(username));
        }
        catch(JetspeedSecurityException e)
        {
            throw new GroupException("Failed to Retrieve User: ", e);
        }
        try
        {
          user.removeGroup(groupName);
          user.update(false);
        }
        catch(Exception e)
        {
            throw new GroupException("Failed to add group info ", e);
        }
    }

    /**
     * Checks for the relationship of user in a group. Returns true when the user is in the given group.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception GroupException when the security provider has a general failure retrieving users.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public boolean inGroup(String username, String groupname)
        throws JetspeedSecurityException
    {
        StringTokenizer st;
        LDAPUser user;

        try
        {
            user = (LDAPUser)JetspeedSecurity.getUser(new UserNamePrincipal(username));
        }
        catch(JetspeedSecurityException e)
        {
            throw new GroupException("Failed to Retrieve User: ", e);
        }

        Vector groups= new Vector();
        try
        {
            for (Enumeration enum = user.getGroupRoles().elements() ;enum.hasMoreElements() ;)
            {
                st = new StringTokenizer((String)enum.nextElement(),",");
                if (st.nextToken().equalsIgnoreCase(groupname))
                {
                    return true;
                }
            }
        }
        catch(Exception e)
        {
            throw new GroupException("Failed to retrieve groups ", e);
        }
        return false;
    }


    /**
     * Retrieves a single <code>Group</code> for a given groupname principal.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param groupname a group principal identity to be retrieved.
     * @return Group the group record retrieved.
     * @exception GroupException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege
     */
    public Group getGroup(String groupname)
        throws JetspeedSecurityException
    {
        BasicAttributes attr= new BasicAttributes();

        LDAPGroup group;
        Vector userurls;
        try
        {
            userurls = JetspeedLDAP.search(JetspeedLDAP.buildURL("ou=groups"),
                                        "(&(uid="+ groupname+")(objectclass=jetspeedgroup))",attrs,true);
          if (userurls.size() == 1){
              return new LDAPGroup((LDAPURL) ((Vector)userurls.elementAt(0)).firstElement());
          }else if(userurls.size() > 1){
            throw new GroupException("Multiple groups with same name");
          }
        }
        catch(Exception e)
        {
            throw new GroupException("Failed to retrieve groups ", e);
        }
        throw new GroupException("Unknown group '" + groupname + "'");

    }


    ///////////////////////////////////////////////////////////////////////////
    // Internal
    ///////////////////////////////////////////////////////////////////////////

    protected JetspeedRunData getRunData()
     {
         JetspeedRunData rundata = null;
         if (this.runDataService != null)
         {
             rundata = this.runDataService.getCurrentRunData();
         }
         return rundata;
     }

    /**
     * Check whether a specified group exists.
     *
     * The login name is used for looking up the account.
     *
     * @param groupName the name of the group to check for existence.
     * @return true if the specified account exists
     * @throws GroupException if there was a general db access error
     *
     */
    protected boolean groupExists(String groupName)
        throws GroupException
    {
        BasicAttributes attr= new BasicAttributes();
        Vector userurls;
        try
        {
            userurls = JetspeedLDAP.search(JetspeedLDAP.buildURL("ou=groups"),
                                        "(&(uid="+ groupName+")(objectclass=jetspeedgroup))",attrs,true);
          if (userurls.size() > 0){
            return true;
          }
        }
        catch(Exception e)
        {
            throw new GroupException("Failed to retrieve groups ", e);
        }
        return false;
    }

    ///////////////////////////////////////////////////////////////////////////
    // Service Init
    ///////////////////////////////////////////////////////////////////////////


    /**
     * This is the early initialization method called by the
     * Turbine <code>Service</code> framework
     * @param conf The <code>ServletConfig</code>
     * @exception throws a <code>InitializationException</code> if the service
     * fails to initialize
     */
    public synchronized void init(ServletConfig conf)
        throws InitializationException
    {
        if (getInit()) return;
        super.init(conf);

        // get configuration parameters from Jetspeed Resources
        ResourceService serviceConf = ((TurbineServices)TurbineServices.getInstance())
                                                     .getResources(JetspeedSecurityService.SERVICE_NAME);

        this.runDataService =
           (JetspeedRunDataService)TurbineServices.getInstance()
               .getService(RunDataService.SERVICE_NAME);

        defaultRole = serviceConf.getString(CONFIG_DEFAULT_ROLE, defaultRole);
        cascadeDelete = serviceConf.getBoolean( CASCADE_DELETE, DEFAULT_CASCADE_DELETE );

        setInit(true);
     }


}