/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security;

import java.util.Iterator;
import java.util.HashMap;

// Junit imports
import junit.framework.Test;
import junit.framework.TestSuite;
import junit.framework.TestCase;


import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

import org.apache.torque.Torque;
import org.apache.torque.TorqueException;

import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.om.security.JetspeedRoleFactory;

import org.apache.jetspeed.services.security.JetspeedSecurityCache;

/**
 * Unit test for RoleManagement interface
 * 
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: TestRoleManagement.java,v 1.4 2002/10/15 17:35:33 taylor Exp $
 */

public class TestRoleManagement extends TestCase {    

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestRoleManagement( String name ) {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[]) 
    {
        junit.awtui.TestRunner.main( new String[] { TestRoleManagement.class.getName() } );
    }
 
    public void setup() 
    {
        //System.out.println("Setup: Testing Turbine Role Management");         
    }

    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite() 
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestRoleManagement.class );
    }

    /**
     * Tests getRoles method
     * @throws Exception
     */

    public void testGetRoles() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;
        HashMap map = new HashMap();

        try
        {
            Iterator roles = service.getRoles();
            while (roles.hasNext())
            {
                role = (Role)roles.next();
                map.put(role.getName(), role);
            }
            assertTrue(map.get("user") != null);
            assertTrue(map.get("admin") != null);
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getRoles Test OK ");

    }

    /**
     * Tests getRoles method
     * @throws Exception
     */

    public void testGetRolesForUser() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;
        HashMap map = new HashMap();

        try
        {
            Iterator roles = service.getRoles("turbine");
            while (roles.hasNext())
            {
                role = (Role)roles.next();
                map.put(role.getName(), role);
                System.out.println("role = " + role.getName());
            }
            assertTrue(map.get("user") != null);
            assertTrue(map.get("admin") == null);

            map.clear();
            roles = service.getRoles("admin");            
            while (roles.hasNext())
            {
                role = (Role)roles.next();
                map.put(role.getName(), role);
                System.out.println("role = " + role.getName());
            }
            assertTrue(map.get("user") != null);
            assertTrue(map.get("admin") != null);
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getRoles Test OK ");

    }

    /**
     * Tests addRole method 
     * @throws Exception
     */

    public void testAddRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            role = JetspeedRoleFactory.getInstance();
            role.setName("bogus");
            service.addRole(role);
            System.out.println("new role id = " + role.getId());
            assertTrue(role.getId() != null);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            role = JetspeedRoleFactory.getInstance();
            role.setName("bogus");
            service.addRole(role);
            fail("Should've thrown a dup key exception on role");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);           
        }

        System.out.println("Completed addRole Test OK ");

    }

    /**
     * Tests getRemoveRole method 
     * @throws Exception
     */

    public void testRemoveRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            service.removeRole("bogus");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.removeRole("catchmeifyoucan");
            fail("Should've thrown a not found exception on role");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);
        }

        System.out.println("Completed addRole Test OK ");

    }

    /**
     * Tests getRole method
     * @throws Exception
     */

    public void testGetRole() throws Exception 
    {
        RoleManagement service = getService();

        try
        {
            Role role = service.getRole("user");
            System.out.println("*** role nm = " + role.getName());
            System.out.println("*** role id = " + role.getId());
            assertTrue(role.getName().equals("user"));
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getRole Test OK ");

    }

    /**
     * Tests saveRole method 
     * @throws Exception
     */

    public void testSaveRole() throws Exception 
    {
        RoleManagement service = getService();

        try
        {
            Role role = service.getRole("user");
            service.saveRole(role);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed saveRole Test OK ");

    }

    /**
     * Tests grantRole method 
     * @throws Exception
     */
    public void testGrantRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            service.grantRole("turbine", "admin");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.grantRole("baduser", "admin");
            fail("Should've thrown a bad user exception on grant");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);           
        }
        try
        {
            service.grantRole("turbine", "badrole");
            fail("Should've thrown a bad role exception on grant");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);           
        }

        System.out.println("Completed grantRole Test OK ");

    }

    /**
     * Tests revokeRole method 
     * @throws Exception
     */
    public void testRevokeRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            service.revokeRole("turbine", "admin");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.revokeRole("baduser", "admin");
            fail("Should've thrown a bad user exception on revoke");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof RoleException);           
        }

        System.out.println("Completed revokeRole Test OK ");

    }

    /**
     * Tests hasRole method 
     * @throws Exception
     */
    public void testHasRole() throws Exception 
    {
        RoleManagement service = getService();
        Role role = null;

        try
        {
            boolean has = service.hasRole("admin", "admin");
            assertTrue(true == has);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            boolean has = service.hasRole("turbine", "admin");
            assertTrue(false == has);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed hasRole Test OK ");

    }

  /*
    Configuration object to run Turbine outside a servlet container
    ( uses turbine.properties )
    */
    private static TurbineConfig config = null;
    
    /**
    Sets up TurbineConfig using the system property:
    <pre>turbine.properties</pre>
    */
    static
    {
        try
        {
            config = new TurbineConfig( "../webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }

    private static RoleManagement getService()
    {
        return (RoleManagement)TurbineServices
                .getInstance()
                .getService(RoleManagement.SERVICE_NAME);
    }

}





