/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */


package org.apache.jetspeed.services.security;

import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.Vector;
import javax.servlet.ServletConfig;

import org.apache.turbine.services.Service;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.services.TurbineBaseService;
import org.apache.turbine.services.InitializationException;
import org.apache.turbine.services.resources.ResourceService;
import org.apache.turbine.util.Log;

import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.om.security.Permission;
import org.apache.jetspeed.services.security.JetspeedSecurityException;

/**
 * The Security Cache Service caches roles and permissions (ACLs)
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: SecurityCacheImpl.java,v 1.5 2002/12/08 19:33:53 taylor Exp $
 */


public class SecurityCacheImpl  extends TurbineBaseService
                                implements SecurityCacheService
{
    protected Map acls = new HashMap();
    protected Map perms = new HashMap();

   /*
    * Utility method for accessing the service
    * implementation
    *
    * @return a SecurityCacheService implementation instance
    */
   protected static SecurityCacheService getService()
   {
       return (SecurityCacheService)TurbineServices
       .getInstance().getService(SecurityCacheService.SERVICE_NAME);
   }

   /*
    * Load the security cache for the given user's roles and permissions.
    *
    * @param JetspeedUser the user to cache all role and permission information for.
    */
    public void load(String username)
        throws JetspeedSecurityException
    {                
        CachedAcl acl = new CachedAcl(username);
        acl.setRoles(JetspeedSecurity.getRoles(username));
        acls.put(username, acl);
        if (perms.size() == 0)
        {
            loadRolePermissions();
        }
    }

    public void unload(String username)
    {
        acls.remove(username);
    }

    public Role getRole(String roleName)
    {
        return (Role)perms.get(roleName);
    }

    public Role getRole(String username, String roleName)
    {
        CachedAcl acl = (CachedAcl)acls.get(username);
        if (acl == null)
        {
            return null;
        }
        return acl.getRole(roleName);
    }
    
    public void addRole(Role role)
    {
        if (!perms.containsKey(role.getName()))
        {
            perms.put(role.getName(), new HashMap());
        }        
    }

    public void addRole(String username, Role role)
    {
        CachedAcl acl = (CachedAcl)acls.get(username);
        if (null != acl)
        {
            acl.addRole(role);
        }
        perms.put(role.getName(), new HashMap());

    }

    public boolean hasRole(String username, String roleName)
    {
        CachedAcl acl = (CachedAcl)acls.get(username);
        if (null != acl)
        {
            return acl.hasRole(roleName);
        }
        return false;
    }


    public void removeRole(String username, String roleName)
    {
        CachedAcl acl = (CachedAcl)acls.get(username);
        if (null != acl)
        {
            acl.removeRole(roleName);
        }
        perms.remove(roleName);
    }

    public CachedAcl getAcl(String username)
    {
        return (CachedAcl)acls.get(username);
    }


    public Iterator getRoles(String username)
    {
        CachedAcl acl = (CachedAcl)acls.get(username);
        if (null != acl)
        {
            return acl.getRoles();
        }
        return null;        
    }

    public Permission getPermission(String roleName, String permissionName)
    {        
        Map map = (Map)perms.get(roleName);
        if (null != map)
        {
            return (Permission)map.get(permissionName);
        }
        return null;        
    }
    
    public void addPermission(String roleName, Permission permission)
    {
        Map map = (Map)perms.get(roleName);
        if (null != map)
        {
            map.put(permission.getName(), permission);
        }
    }

    public boolean hasPermission(String roleName, String permissionName)
    {
        Map map = (Map)perms.get(roleName);
        if (null != map)
        {
            return map.containsKey(permissionName);
        }
        return false;
    }

    public void removePermission(String roleName, String permissionName)
    {
        Map map = (Map)perms.get(roleName);
        if (null != map)
        {
            map.remove(permissionName);
        }
    }

    public Iterator getPermissions(String roleName)
    {
        Map map = (Map)perms.get(roleName);
        if (map != null)
        {
            return map.values().iterator();
        }
        return null;
    }


    public void removeAllRoles(String rolename)
    {
        Iterator iterator = acls.values().iterator();
        while (iterator.hasNext())
        {
            CachedAcl acl = (CachedAcl)iterator.next();
            acl.removeRole(rolename);
        }
        perms.remove(rolename);
    }

    public void removeAllPermissions(String permissionName)
    {
        Iterator iterator = perms.values().iterator();
        while (iterator.hasNext())
        {
            Map map = (Map)iterator.next();
            map.remove(permissionName);
        }
    }

    public void loadRolePermissions()
    {
        try
        {
            Iterator roles = JetspeedSecurity.getRoles();
            while (roles.hasNext())
            {
                Role role = (Role)roles.next();
                Map map = new HashMap();
                Iterator prms = JetspeedSecurity.getPermissions(role.getName());
                while (prms.hasNext())
                {
                    Permission perm = (Permission)prms.next();
                    map.put(perm.getName(), perm);
                }
                perms.put(role.getName(), map);
            }
        }
        catch (JetspeedSecurityException e)
        {
            Log.error(e);
        }
    }
    ///////////////////////////////////////////////////////////////////////////
    // Service Init
    ///////////////////////////////////////////////////////////////////////////


    /**
     * This is the early initialization method called by the 
     * Turbine <code>Service</code> framework
     * @param conf The <code>ServletConfig</code>
     * @exception throws a <code>InitializationException</code> if the service
     * fails to initialize
     */
    public synchronized void init(ServletConfig conf) 
        throws InitializationException 
    {
        if (getInit()) return;

        super.init(conf);

        // get configuration parameters from Jetspeed Resources
        ResourceService serviceConf = ((TurbineServices)TurbineServices.getInstance())
                                                     .getResources(SecurityCacheService.SERVICE_NAME);


        setInit(true);
     }

}


