/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security;

import java.util.Iterator;

import org.apache.turbine.services.Service;


// Turbine
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.services.Service;


import org.apache.jetspeed.om.security.Permission;

/**
 * <p> The <code>PermissionManagement</code> interface describes contract between 
 * the portal and security provider required for Jetspeed Permission Management.
 * This interface enables an application to be independent of the underlying 
 * permission management technology.
 *
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: JetspeedPermissionManagement.java,v 1.2 2002/06/28 05:37:33 taylor Exp $
 */

public abstract class JetspeedPermissionManagement
{
    public String SERVICE_NAME = "PermissionManagement";

    /*
     * Utility method for accessing the service
     * implementation
     *
     * @return a PermissionManagement implementation instance
     */
    protected static PermissionManagement getService()
    {
        return (PermissionManagement)TurbineServices
        .getInstance().getService(PermissionManagement.SERVICE_NAME);
    }

    /**
     * Retrieves all <code>Permission</code>s for a given rolename principal.
     *   
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param rolename a role name identity to be retrieved.
     * @return Iterator over all permissions associated to the role principal.
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    public static Iterator getPermissions(String rolename)
        throws JetspeedSecurityException
    {
        return getService().getPermissions(rolename);
    }

    /**
     * Retrieves all <code>Permission</code>s.
     *   
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @return Iterator over all permissions.
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    public static Iterator getPermissions()
        throws JetspeedSecurityException
    {
        return getService().getPermissions();
    }

    /**
     * Adds a <code>Permission</code> into permanent storage. 
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    public static void addPermission(Permission permission)
        throws JetspeedSecurityException
    {
        getService().addPermission(permission);
    }

    /**
     * Saves a <code>Permission</code> into permanent storage. 
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    public static void savePermission(Permission permission)
        throws JetspeedSecurityException
    {
        getService().savePermission(permission);
    }

    /**
     * Removes a <code>Permission</code> from the permanent store.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param permissionName the principal identity of the permission to be retrieved.
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    public static void removePermission(String permissionName)
        throws JetspeedSecurityException
    {
        getService().removePermission(permissionName);
    }

    /**
     * Grants a permission to a role. 
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param roleName grant a permission to this role.
     * @param permissionName the permission to grant to the role.
     * @exception PermissionException when the security provider has a general failure retrieving permissions.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    public static void grantPermission(String roleName, String permissionName)
        throws JetspeedSecurityException
    {
        getService().grantPermission(roleName, permissionName);
    }

    /**
     * Revokes a permission from a role. 
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param roleName grant a permission to this role.
     * @param permissionName the permission to grant to the role.     
     * @exception PermissionException when the security provider has a general failure retrieving permissions.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    public static void revokePermission(String roleName, String permissionName)
        throws JetspeedSecurityException
    {
        getService().revokePermission(roleName,permissionName);
    }

    /**
     * Checks for the relationship of role has a permission. Returns true when the role has the given permission.
     *
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param roleName grant a permission to this role.
     * @param permissionName the permission to grant to the role.    
     * @exception PermissionException when the security provider has a general failure retrieving permissions.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    public static boolean hasPermission(String roleName, String permissionName)
        throws JetspeedSecurityException
    {
        return getService().hasPermission(roleName,permissionName);
    }

    /**
     * Retrieves a single <code>Permission</code> for a given permissionName principal.
     *   
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param permissionName a permission principal identity to be retrieved.
     * @return Permission the permission record retrieved.
     * @exception PermissionException when the security provider has a general failure.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    public static Permission getPermission(String permissionName)
        throws JetspeedSecurityException
    {
        return getService().getPermission(permissionName);
    }

}











