/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portal.portlets;

// Turbine stuff
import org.apache.turbine.util.RunData;
import org.apache.turbine.services.velocity.TurbineVelocity;
import org.apache.turbine.services.pull.TurbinePull;
import org.apache.turbine.modules.ActionLoader;

// Jetspeed stuff
import org.apache.jetspeed.portal.Portlet;
import org.apache.jetspeed.portal.portlets.AbstractPortlet;
import org.apache.jetspeed.portal.PortletException;
import org.apache.jetspeed.portal.PortletConfig;
import org.apache.jetspeed.services.TemplateLocator;
import org.apache.jetspeed.util.template.JetspeedLink;
import org.apache.jetspeed.util.template.JetspeedTemplateLink;
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.security.JetspeedSecurityService;


// Ecs stuff
import org.apache.ecs.ConcreteElement;
import org.apache.jetspeed.util.JetspeedClearElement;

// Velocity Stuff
import org.apache.velocity.context.Context;
import org.apache.turbine.util.Log;

/**
 * A Velocity based portlet implementation
 * 
 * @author <a href="mailto:re_carrasco@bco011.sonda.cl">Roberto Carrasco</a>
 * @author <a href="mailto:raphael@apache.org">Raphal Luta</a>
 * @author <a href="mailto:paulsp@apache.org">Paul Spencer</a>
 */
public class VelocityPortlet extends AbstractInstancePortlet

{

    /** 
    How long in milliseconds will the content of this portlet be cached,
    that is, how long before regenerating it.
    The default is zero - never cache the content.
    This is specified in the portlet config parameter "cache-period-milliseconds"
    **/
    private long cachePeriod = 0;

    /**
     * Initialize this portlet by looking up the cache period (default is zero)
     * @throws PortletException Initialization failed
     */
    public void init() throws PortletException {

        PortletConfig config = this.getPortletConfig();

        try
        {
            cachePeriod = Long.parseLong( config.getInitParameter("cache-period-milliseconds","0") );
        } catch (NumberFormatException e) {
            cachePeriod = 0;
        }
    }

    /**
    This methods outputs the content of the portlet for a given
    request.

    @param data the RunData object for the request
    @return the content to be displayed to the user-agent
    */
    public ConcreteElement getContent( RunData data )
    {
        //if caching is turned off or no expiration time set, generate and return the content
        if (cachePeriod == 0 || null == getExpirationMillis())
            return getVelocityContent(data);

        //is the cached content still valid, if not, generate and return the content
        if (getExpirationMillis().longValue() <= System.currentTimeMillis())
            return getVelocityContent(data);

        //else, the cached content is fine to be returned
        return getContent( data, null , true );
    }


    /**
     * @param rundata The RunData object for the current request
     */    
    public ConcreteElement getVelocityContent( RunData rundata )
    {
        Log.debug("VelocityPortlet::getVelocityContent");

        // create a blank context and with all the global application
        // Pull Tools inside
        Context context = TurbineVelocity.getContext();
        context.put( "data", rundata );
        context.put( "portlet", this );
        context.put( "conf", this.getPortletConfig() );
        context.put( "skin", this.getPortletConfig().getPortletSkin() );

        //add a tool for security information to the template.
// FIXME:  This is causing an Exception.  Is it needed?         
//        context.put( "security", (JetspeedSecurityService)JetspeedSecurity.getService() );
        
        String template = getPortletConfig().getInitParameter("template");
        
        if (template != null)
        {
            context.put("template",template); 
        }

        // Put the request and session based contexts
        TurbinePull.populateContext(context, rundata);
        
        //We need to initialize the jlink tool with ourselves to
        // enable links between portlets.

        // FIXME: the tool jlink is deprecated.
        Object jlink = context.get( "jlink" );
        if (jlink instanceof JetspeedTemplateLink )
            ((JetspeedTemplateLink)jlink).setPortlet( this );

        Object jslink = context.get( "jslink" );
        if (jslink instanceof JetspeedLink )
            ((JetspeedLink)jslink).setPortlet( this );

        String actionName = getPortletConfig().getInitParameter("action");
        
        if (actionName != null)
        {
            // store the context so that the action can retrieve it
            Log.debug("VelocityPortlet found action "+actionName+" context "+context);
            rundata.getTemplateInfo().setTemplateContext( "VelocityPortletContext", context );

            // if there is an action with the same name in modules/actions/portlets exec it
            try
            {
                ActionLoader.getInstance().exec( rundata, actionName );
            }
            catch( Exception e)
            {
               Log.error( e );
            }
        }
 
        // either the action selected the template, or use the default template 
        // defined in the registry
        if (context.get( "template" )!=null)
        {
            template = (String)context.get( "template" );
        }
        
        // generate the content
        JetspeedClearElement element = null;

        try
        {
            if (-1 == template.indexOf(".vm"))
                template = template + ".vm";
            String templatePath = TemplateLocator.locatePortletTemplate(rundata, template);
            if (cachePeriod > 0)
            {
                String s = TurbineVelocity.handleRequest(context, templatePath);
                setExpirationMillis(cachePeriod + System.currentTimeMillis());
                element = new JetspeedClearElement( s );
                clearContent();  // doing this because setContent() is not overwriting current content.
                setContent(element);
            } else {
                TurbineVelocity.handleRequest(context, templatePath, rundata.getOut());
            }
        }
        catch( Exception e)
        {
            element = new JetspeedClearElement( e.toString() );
        }
        
        TurbineVelocity.requestFinished(context);
        
        if (element == null) element = new JetspeedClearElement("");

        return element;
    }


}

