/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.om.registry.base;

import org.apache.jetspeed.om.registry.MediaTypeRegistry;
import org.apache.jetspeed.om.registry.MediaTypeEntry;
import org.apache.jetspeed.om.registry.RegistryEntry;
import org.apache.jetspeed.om.registry.InvalidEntryException;
import org.apache.jetspeed.om.registry.RegistryException;
import org.apache.jetspeed.services.Registry;
import org.apache.jetspeed.capability.CapabilityMap;

import org.apache.turbine.util.Log;

import java.util.Iterator;
import java.util.Enumeration;
import java.util.List;
import java.util.ArrayList;

/**
 * Extends BaseRegistry implementation to override object creation
 * method
 *
 * @author <a href="mailto:raphael@apache.org">Raphal Luta</a>
 * @version $Id: BaseMediaTypeRegistry.java,v 1.4 2002/11/07 20:17:54 raphael Exp $
 */
public class BaseMediaTypeRegistry extends BaseOrderedRegistry
        implements MediaTypeRegistry
{

    /*
     * Find media-types in this registry that match the CapabilityMap requirements
     *
     * @param category The category and optional subcategories.
     * @return Iterator The result as an iterator.
     */
    public Iterator findForCapability(CapabilityMap cm)
    {
        if (cm == null)
        {
            return null;
        }

        String type = cm.getPreferredType().getContentType();
        List result = new ArrayList();

        if (DEBUG && Log.getLogger().isDebugEnabled())
        {
            Log.debug("MediaTypeRegistry: looking for type " + type );
        }

        if (type == null)
        {
            return result.iterator();
        }

        try
        {
            Enumeration en = getEntries();
            while(en.hasMoreElements())
            {
                MediaTypeEntry mte = (MediaTypeEntry)en.nextElement();

                if (DEBUG && Log.getLogger().isDebugEnabled())
                {
                    Log.debug("MediaTypeRegistry: found MediaTypeEntry for type " + mte.getMimeType() );
                }
                if (type.equals(mte.getMimeType()))
                {
                    result.add(mte);
                }
            }
        }
        catch(Exception e)
        {
            Log.error(e);
        }

        if (DEBUG && Log.getLogger().isDebugEnabled())
        {
            Log.debug("MediaTypeRegistry: found " + result.size() + " entries." );
        }

        return result.iterator();

    }

    /**
    @see Registry#setEntry
    */
    public void setEntry( RegistryEntry entry ) throws InvalidEntryException
    {
        // Delegate to the RegistryService to ensure correct handling of
        // persistence if using file fragments

        try
        {
            Registry.addEntry(Registry.MEDIA_TYPE, entry);
        }
        catch (RegistryException e)
        {
            Log.error(e);
        }
    }

    /**
    @see Registry#addEntry
    */
    public void addEntry( RegistryEntry entry ) throws InvalidEntryException
    {
        // Delegate to the RegistryService to ensure correct handling of
        // persistence if using file fragments

        try
        {
            Registry.addEntry(Registry.MEDIA_TYPE, entry);
        }
        catch (RegistryException e)
        {
            Log.error(e);
        }
    }

    /**
    @see Registry#removeEntry
    */
    public void removeEntry( String name )
    {
        // Delegate to the RegistryService to ensure correct handling of
        // persistence if using file fragments

        Registry.removeEntry(Registry.MEDIA_TYPE, name);
    }

    /**
    @see Registry#removeEntry
    */
    public void removeEntry( RegistryEntry entry )
    {
        // Delegate to the RegistryService to ensure correct handling of
        // persistence if using file fragments

        if (entry != null)
        {
            Registry.removeEntry(Registry.MEDIA_TYPE, entry.getName());
        }
    }

    /**
     * Creates a new RegistryEntry instance compatible with the current
     * Registry instance implementation
     *
     * @return the newly created RegistryEntry
     */
    public RegistryEntry createEntry()
    {
        return new BaseMediaTypeEntry();
    }
}
