/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/core/ActionContributorHelper.java,v 1.8 2007/10/25 03:16:49 kohnosuke Exp $
 * $Revision: 1.8 $
 * $Date: 2007/10/25 03:16:49 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.core;

import jp.sourceforge.jeextension.styleeditor.StyleEditorPlugin;
import jp.sourceforge.jeextension.styleeditor.util.ProxyFactory;
import jp.sourceforge.jeextension.styleeditor.util.ReflectFactory;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.AbstractDocument;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentPartitioner;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.IVerticalRulerInfo;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.texteditor.ITextEditor;

/**
 * ActionContributorHelper
 * <!-- #IntID:00011 support java editor for cvs repository view -->
 * @since 2.0.0
 */
class ActionContributorHelper implements IPropertyChangeListener {
    
    /** ISourceViewerActivator */
    private final ISourceViewerActivator fActivator;
    
    /**
     * Constructor.
     * @param activator ISourceViewerActivator
     */
    ActionContributorHelper(ISourceViewerActivator activator) {
        fActivator = activator;
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.util.IPropertyChangeListener#propertyChange(org.eclipse.jface.util.PropertyChangeEvent)
     */
    public void propertyChange(PropertyChangeEvent event) {
        
        StyleEditorCore core = StyleEditorPlugin.getDefault().getCore();
        IEditorPart part = core.getActiveEditorPart(this);
        if (part != null && part.getEditorInput() != null) {
            activateSourceViewer(part, true);
        }
    }
    
    /**
     * Setup Jeex Style Editor styles to IEditorPart.
     * @param part  IEditorPart
     */
    void setActiveEditorPart(IEditorPart part) {
        
        // regist editor
        StyleEditorCore core = StyleEditorPlugin.getDefault().getCore();
        if (part == null) {
            core.removeActiveEditorPart(this);
        }
        else {
            
            core.setActiveEditorPart(this, part);
            
            // set PreferenceStore
            IPreferenceStore prefStore =
                StyleEditorPlugin.getDefault().getPreferenceStore();
            prefStore.addPropertyChangeListener(this);
            
            activateSourceViewer(part, false);
            
            // set ruler double click action
            if (part instanceof ITextEditor) {
                ITextEditor editor = (ITextEditor) part;
                setupRulerDoubleClickAction(editor);
            }
        }
    }
    
    /**
     * Setup RulerDoubleClick's Action.
     * @param editor    ITextEditor
     */
    private void setupRulerDoubleClickAction(ITextEditor editor) {
        IVerticalRulerInfo rulerInfo = (IVerticalRulerInfo)
        editor.getAdapter(IVerticalRulerInfo.class);
        IAction action;
        
        String version = StyleEditorPlugin.getRuntimeVersion();
        
        // 3.0.2 higher
        if (version.compareTo("3.0.2") > 0) {
            ReflectFactory factory = new ReflectFactory();
            factory.setClass("org.eclipse.debug.ui.actions."
                    + "ToggleBreakpointAction");
            factory.setType(new Class[]{
                    IWorkbenchPart.class,
                    IDocument.class,
                    IVerticalRulerInfo.class,
            });
            factory.setParam(new Object[] {
                    editor,
                    null,
                    rulerInfo,
            });
            action = (IAction) factory.newInstance();
        }
        // 3.0.2 equal or lower
        else {
            ReflectFactory factory = new ReflectFactory();
            factory.setClass("org.eclipse.jdt.internal.debug.ui.actions."
                    + "ManageBreakpointRulerAction");
            factory.setType(new Class[]{
                    IVerticalRulerInfo.class,
                    ITextEditor.class,
            });
            factory.setParam(new Object[] {
                    rulerInfo,
                    editor,
            });
            action = (IAction) factory.newInstance();
        }
        
        editor.setAction("RulerDoubleClick", action);
    }
    
    /**
     * Activate Style Editor SourceViewer.
     * @param part  IEditorPart
     * @param force force reconfigure
     */
    private void activateSourceViewer(IEditorPart part, boolean force) {
        
        if (part == null) {
            return;
        }
        // get ISourceViewer
        ReflectFactory factory = new ReflectFactory();
        factory.setTarget(part);
        factory.setClass("org.eclipse.ui.texteditor.AbstractTextEditor");
        factory.setName("getSourceViewer");
        
        ISourceViewer viewer = (ISourceViewer) factory.invoke();
        if (viewer == null) {
            return;
        }
        
        StyleEditorCore core = StyleEditorPlugin.getDefault().getCore();
        Object config = core.getActiveViewerConfig(viewer);
        if (config == null || force) {
            
            // resolve file extension
            String fileExtension = getFileExtension(part);
            
            // activate font
            viewer.getTextWidget().setFont(
                    StyleEditorTools.getFont(fileExtension));
            
            // setup IDocument, IDocumentPartitioner
            StyleEditorTools tools =
                StyleEditorPlugin.getDefault().getEditorTools();
            IDocument document;
            IDocumentPartitioner partitioner;
            document = viewer.getDocument();
            partitioner = tools.createDocumentPartitioner(fileExtension);
            partitioner.connect(document);
            
            if (document instanceof AbstractDocument) {
                AbstractDocument doc = (AbstractDocument) document;
                IDocumentPartitioner partitioner2 =
                    doc.getDocumentPartitioner("___java_partitioning");
                if (partitioner2 != null) {
                    doc.setDocumentPartitioner(
                            "___java_partitioning", partitioner);
                }
                else {
                    doc.setDocumentPartitioner(partitioner);
                }
            }
            else {
                document.setDocumentPartitioner(partitioner);
            }
            
            // viewer configure
            Class configCls = getSourceViewerConfiguration(part);
            fActivator.activate(viewer, document, configCls, part);
        }
    }
    
    /**
     * Return SourceViewerConfiguration Dynamic Proxy class.
     * 
     * @param part  target editor instance
     * @return  Dynamic Proxy class.
     */
    private Class getSourceViewerConfiguration(IEditorPart part) {
        
        // get SourceViewerConfiguration
        ReflectFactory f = new ReflectFactory();
        f.setTarget(part);
        f.setClass("org.eclipse.ui.texteditor.AbstractTextEditor");
        f.setName("getSourceViewerConfiguration");
        Object sourceViewerConfig = f.invoke();
        
        // resolve target file extension
        String fileExtension = getFileExtension(part);
        
        // create ProxyMethodHandler
        SourceViewerConfigurationProxyHandler handler =
            new SourceViewerConfigurationProxyHandler(sourceViewerConfig);
        handler.setFileExtension(fileExtension);
        
        // create dynamic proxy class
        final ClassLoader loader = getClass().getClassLoader();
        Class target = null;
        try {
            ProxyFactory proxyFactory = new ProxyFactory(loader);
            proxyFactory.setSuperclass(sourceViewerConfig.getClass());
            proxyFactory.setHandler(handler);
            target = proxyFactory.createClass();
        }
        catch (Exception e) {
            StyleEditorPlugin.log(e);
        }
        return target;
    }
    
    /**
     * Return file's extension.
     * @param part  IEditorPart
     * @return  File extension
     */
    private String getFileExtension(IEditorPart part) {
        String fileExtension;
        IEditorInput input = part.getEditorInput();
        String fileName = input.getName();
        if (fileName.indexOf('.') != -1) {
            fileExtension = fileName.substring(
                    fileName.indexOf('.') + 1, fileName.length());
            // #IntID:00011 support java editor for cvs repository view
            if (fileExtension.indexOf(' ') != -1) {
                fileExtension = fileExtension.substring(0, 
                        fileExtension.indexOf(' '));
            }
        }
        else {
            fileExtension = "*";
        }
        return fileExtension;
    }
}
