/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/preferences/PartitionEntryDialog.java,v 1.1 2005/01/29 03:24:51 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2005/01/29 03:24:51 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.preferences;

import jp.sourceforge.jeextension.common.plugin.DefaultDialog;
import jp.sourceforge.jeextension.styleeditor.StyleEditorPlugin;
import jp.sourceforge.jeextension.styleeditor.core.ColorManager;
import jp.sourceforge.jeextension.styleeditor.core.EditorTools;
import jp.sourceforge.jeextension.styleeditor.entry.DefinedStyleEntry;
import jp.sourceforge.jeextension.styleeditor.entry.IPartitionEntry;
import jp.sourceforge.jeextension.styleeditor.entry.PartitionEntry;
import jp.sourceforge.jeextension.styleeditor.entry.PartitionEntryList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.TableColumn;

/**
 * PartitionEntryDialog
 */
class PartitionEntryDialog extends DefaultDialog {
    
    /** Add button. */
    private Button fButtonAdd;
    
    /** Remove button. */
    private Button fButtonDel;
    
    /** Down button. */
    private Button fButtonDn;
    
    /** Modify button. */
    private Button fButtonMod;
    
    /** Up button. */
    private Button fButtonUp;
    
    /** ColorManager for this Dialog. */
    private ColorManager fColorManager = new ColorManager();
    
    /** partition entry table. */
    private Table fTable;
    
    /** current editting extension. */
    private String fExtension;
    
    /**
     * Initiate this dialog page.
     * @param shell parent shell.
     * @param ext   extension name. (ex. java, means *.java)
     */
    PartitionEntryDialog(Shell shell, String ext) {
        super(shell);
        fExtension = ext;
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.dialogs.Dialog#close()
     */
    public boolean close() {
        this.fColorManager.dispose();
        return super.close();
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
     */
    public void handleEvent(Event event) {
        if (event.widget == fButtonAdd) {
            fireAddEntry();
        }
        if (event.widget == fButtonMod) {
            fireModifyEntry();
        }
        if (event.widget == fButtonDel) {
            fireDeleteEntry();
        }
        if (event.widget == fButtonUp) {
            fireMoveUpEntry();
        }
        if (event.widget == fButtonDn) {
            fireMoveDownEntry();
        }
    }
        
    /**
     * Add Objects into target Table.
     * @param   entry   IEntry instance.
     */
    private void addTableItem(IPartitionEntry entry) {
        /* add table item. */
        TableItem item =
                new TableItem(fTable, SWT.NONE);
        item.setText(0, entry.getName());
        if (entry instanceof PartitionEntry) {
            PartitionEntry pe = (PartitionEntry) entry;
            item.setText(1, pe.getStart() + " " + pe.getEnd());
        }
        item.setData(entry);
        item.setForeground(fColorManager.getColor(entry.getRGB()));
    }
    
    /**
     * Create Button.
     * @param composite parent composite.
     * @param str button's string.
     * @return created Button.
     */
    private Button createButton(Composite composite, String str) {
        final Button button = new Button(composite, SWT.NONE);
        button.setLayoutData(new GridData(
          GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_BEGINNING));
        button.setText(str);
        button.addListener(SWT.Selection, this);
        return button;
    }
    
    /**
     * Create TableColumn.
     * @param table parent composite (Table).
     * @param str column's string.
     * @return created TableColumn.
     */
    private TableColumn createTableColumn(Table table, String str) {
        final TableColumn tableColumn = new TableColumn(table, SWT.NONE);
        tableColumn.setWidth(100);
        tableColumn.setText(str);
        return tableColumn;
    }
    
    /**
     * fire Add button action.
     */
    private void fireAddEntry() {
        IPartitionEntry[] container = new IPartitionEntry[1];
        AddPartitionDialog d =
                new AddPartitionDialog(getShell(), container, fExtension);
        if (d.open() == IDialogConstants.OK_ID) {
            // add table item.
            addTableItem(container[0]);
            
            DefinedStyleEntry styleEntry =
                    StyleEditorPlugin.getDefinedStyleEntry();
            
            // add to preference store.
            styleEntry.add(container[0], fExtension);
        }
    }
    
    /**
     * fire Delete button action.
     */
    private void fireDeleteEntry() {
        int sel = fTable.getSelectionIndex();
        if (sel < 0) {
            return;
        }
        TableItem item = fTable.getItem(sel);
        IPartitionEntry entry = (IPartitionEntry) item.getData();
        
        // remove from table. and remove from conf list.
        fTable.remove(sel);
        StyleEditorPlugin.getDefinedStyleEntry().remove(entry, fExtension);
    }
    
    /**
     * fire Modify button action.
     */
    private void fireModifyEntry() {
        int sel = fTable.getSelectionIndex();
        if (sel < 0) {
            return;
        }
        TableItem item = fTable.getItem(sel);
        IPartitionEntry entry = (IPartitionEntry) item.getData();
        IPartitionEntry[] container = new IPartitionEntry[1];
        container[0] = entry;
        AddPartitionDialog d =
                new AddPartitionDialog(getShell(), container, fExtension);
        if (d.open() == IDialogConstants.OK_ID) {
                
            item.setText(0, entry.getName());
            if (entry instanceof PartitionEntry) {
                PartitionEntry pe = (PartitionEntry) entry;
                item.setText(1, pe.getStart() + " " + pe.getEnd());
            }
            item.setData(entry);
            item.setForeground(fColorManager.getColor(entry.getRGB()));
        }
    }
    
    /**
     * fire Down button action.
     */
    private void fireMoveDownEntry() {
        fireMoveEntry(1);
    }

    /**
     * fire Up button action.
     */
    private void fireMoveUpEntry() {
        fireMoveEntry(-1);
    }

    /**
     * fire Up/Down button action.
     * @param increment Up=-1 / Down=1
     */
    private void fireMoveEntry(int increment) {
        int sel = fTable.getSelectionIndex();
        if (sel < 0) {
            return;
        }
        Object obj = fTable.getSelection()[0].getData();
            
        PartitionEntryList list =
                EditorTools.getPartitionEntryList(fExtension, false);
        int index = list.indexOf(obj);
        Object removeObj = list.remove(index);
        list.add(index + increment, removeObj);
            
        refleshTableEntry();
        fTable.setSelection(sel + increment);
        refleshButtonEnable();
    }
    
    /**
     * Change button press enabled when table entry change.
     */
    private void refleshButtonEnable() {
        int sel = fTable.getSelectionIndex();
        if (fTable.getItemCount() == 1) {
            fButtonUp.setEnabled(false);
            fButtonDn.setEnabled(false);
        }
        else
        if (sel == 0) {
            fButtonUp.setEnabled(false);
            fButtonDn.setEnabled(true);
        }
        else
        if (sel == fTable.getItemCount() - 1) {
            fButtonUp.setEnabled(true);
            fButtonDn.setEnabled(false);
        }
        else {
            fButtonUp.setEnabled(true);
            fButtonDn.setEnabled(true);
        }
    }
        
    /**
     * Reflesh table entries.
     */
    private void refleshTableEntry() {
        // clear all table content.
        fTable.removeAll();
        
        // add entry depends current partition to table.
        PartitionEntryList entryList =
                EditorTools.getPartitionEntryList(fExtension, false);
        for (int i = 0; i < entryList.size(); i++) {
            IPartitionEntry entry = entryList.getPartitionEntry(i);
            addTableItem(entry);
        }
    }

    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
     */
    protected Control createDialogArea(Composite parent) {
        
        final Composite composite = (Composite) super.createDialogArea(parent);
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        final GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        composite.setLayout(gridLayout);
        
        fTable = new Table(composite, SWT.FULL_SELECTION | SWT.BORDER);
        final GridData gridData = new GridData(GridData.FILL_BOTH);
        gridData.verticalSpan = 5;
        fTable.setLayoutData(gridData);
        fTable.setLinesVisible(true);
        fTable.setHeaderVisible(true);
        
        createTableColumn(fTable, getMessage("pref.com.colmn.type"));
        createTableColumn(fTable, getMessage("pref.com.colmn.rule"));
        
        fButtonAdd = createButton(composite, getMessage("pref.com.btn.add"));
        fButtonMod = createButton(composite, getMessage("pref.com.btn.mod"));
        fButtonDel = createButton(composite, getMessage("pref.com.btn.del"));
        fButtonUp  = createButton(composite, getMessage("pref.com.btn.up"));
        fButtonDn  = createButton(composite, getMessage("pref.com.btn.dn"));
        
        // entry data.
        refleshTableEntry();
        
        return composite;
    }

    /**
     * Adds buttons to this dialog's button bar.
     * <p>
     * The <code>Dialog</code> implementation of this framework method adds 
     * standard ok and cancel buttons using the <code>createButton</code>
     * framework method. These standard buttons will be accessible from 
     * <code>getCancelButton</code>, and <code>getOKButton</code>.
     * Subclasses may override.
     * </p>
     *
     * @param parent the button bar composite
     */
    protected void createButtonsForButtonBar(Composite parent) {
        // create OK button only
        createButton(
            parent,
            IDialogConstants.OK_ID,
            IDialogConstants.OK_LABEL,
            true);
    }
}
