/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.commons/src/jp/sourceforge/jeextension/common/xml/XMLWriter.java,v 1.1 2004/12/17 02:25:24 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2004/12/17 02:25:24 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.common.xml;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXResult;

import org.apache.xml.serializer.SerializerBase;
import org.apache.xml.serializer.ToXMLStream;
import org.w3c.dom.Document;


/**
 * XMLWriter is SAX based XML Document writer.
 */
public final class XMLWriter {
    
    /** Writer encoding. */
    private String fileEncoding;
    
    /**
     * Initiate XMLWriter using default encoding.
     */
    public XMLWriter() {
        this(System.getProperty("file.encoding"));
    }
    
    /**
     * Initiate XMLWriter using encoding parameter.
     * @param encoding  file encoding.
     */
    public XMLWriter(String encoding) {
        super();
        this.fileEncoding = encoding;
    }

    /**
     * Write XMLNode object to file as xml document file.
     * @param xmlFile   target xml file object.
     * @param xmlNode   wrote XMLNode.
     * @throws XMLException throws when writer failed to write to xml file. or
     *                      arguments are null.
     */
    public void write(File xmlFile, XMLNode xmlNode) throws XMLException {

        try {
            XMLDocumentBuilder builder = new XMLDocumentBuilder(xmlNode);
            Document document = builder.build();
            write(xmlFile, document);
        }
        catch (Exception e) {
            throw new XMLException(e);
        }
    }
    
    /**
     * Write org.w3c.Document to file as xml document file.
     * @param xmlFile   target xml file object.
     * @param doc       wrote org.w3c.Document
     * @throws XMLException throws when writer failed to write to xml file. or
     *                      arguments are null.
     */
    public void write(File xmlFile, Document doc) throws XMLException {
        try {
            String dtd = getDTD(doc);
            SerializerBase contentHandler = getSerializerBase(xmlFile, dtd);

            // ready Transformer object for writing.
            Transformer transformer = getTransformer();
            transformer.transform(new DOMSource(doc),
                new SAXResult(contentHandler));
            
            // close writer stream.
            contentHandler.getWriter().close();
        }
        catch (Exception e) {
            throw new XMLException(
                "writer failed to write into file. xmlFile="
                + xmlFile.getPath() + " doc=" + doc, e);
        }
    }

    /**
     * Returns org.apache.xml.serializer.SerializerBase object implement org.
     * xml.sax.ContentHandler that layouting indent 4.
     * @param file  target File object.
     * @param dtd   DTD's path for target xml document.
     * @return  org.apache.xml.serializer.SerializerBase implement
     *          org.xml.sax.ContentHandler
     * @throws IOException throws when failed to output to the file.
     */
    private SerializerBase getSerializerBase(File file, String dtd)
            throws IOException {
        OutputStream   out    = new FileOutputStream(file);
        Writer         writer = new OutputStreamWriter(out, fileEncoding);
        SerializerBase base   = new ToXMLStream();
        base.setWriter(writer);
        base.setIndent(true);
        base.setIndentAmount(4);
        base.setEncoding(fileEncoding);
        if (dtd != null) {
            base.setDoctypeSystem(dtd);
        }
        return base;
    }

    /**
     * Returns javax.xml.transform.Transformer object.
     * @return javax.xml.transform.Transformer
     * @throws TransformerException throw whenTransformer object creation
     *                              fails.
     */
    private Transformer getTransformer() throws TransformerException {
        TransformerFactory factory     = TransformerFactory.newInstance();
        Transformer        transformer = factory.newTransformer();
        return transformer;
    }
    
    /**
     * Returns DTD (file name) for Documents.
     * @param doc   Document object.
     * @return String means dtd. (file name)
     */
    private String getDTD(Document doc) {
        String documentNodeName = doc.getDocumentElement().getNodeName();
        if (documentNodeName.equals("xxxx")) {
//            return "XXXx.dtd";
        }
        return null;
    }
}
