/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.commons/src/jp/sourceforge/jeextension/common/xml/XMLReader.java,v 1.1 2004/12/17 02:25:24 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2004/12/17 02:25:24 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.common.xml;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.xml.sax.InputSource;


/**
 * XMLReader is based DOM XML file reader.
 */
public final class XMLReader {
    
    /** is using parser to doing validation process. */
    private boolean parseValidation = false;
    
    /** Reader encoding. */
    private String fileEncoding;
    
    /**
     * Initiate XMLXMLReader using default encoding.
     */
    public XMLReader() {
        this(System.getProperty("file.encoding"));
    }
    
    /**
     * Initiate XMLReader using encoding parameter.
     * @param encoding  file encoding.
     */
    public XMLReader(String encoding) {
        super();
        this.fileEncoding = encoding;
    }

    /**
     * Read xml file based on DOM.
     * Returns XMLNode that means to be xml file.
     * XMLNode's structure reference. {@link XMLNode}
     * @param xmlFile   XML File object.
     * @return XMLNode Object.
     * @throws XMLException throw when reader failed to read files.
     */
    public XMLNode readAsXMLNode(File xmlFile) throws XMLException {

        Document document = readAsDocument(xmlFile);
        
        // transform Document to XMLNode object.
        XMLNodeBuilder builder = new XMLNodeBuilder(document);
        return builder.build();
    }

    /**
     * Read xml file reading stream for xml document based on DOM.
     * Returns XMLNode that means to be xml file.
     * XMLNode's structure reference. {@link XMLNode}
     *
     * @param stream    java.io.InputStream.
     * @return XMLNode Object.
     * @throws XMLException throw when reader failed to read files.
     */
    public XMLNode readAsXMLNode(InputStream stream) throws XMLException {

        Document document = readAsDocument(stream);
        
        // tramsform Document to XMLNode object.
        XMLNodeBuilder builder = new XMLNodeBuilder(document);
        return builder.build();
    }
    
    /**
     * Read argument's stream as XML document based on DOM.
     * Returns org.w3c.dom.Document object that be reading result.
     * 
     * @param stream    java.io.InputStream
     * @return org.w3c.dom.Document Object.
     * @throws XMLException throw when reader failed to read files.
     */
    public Document readAsDocument(InputStream stream) throws XMLException {

        try {
            // Create InputSource as target XML document.
            InputSource input =
                    new InputSource(
                            new InputStreamReader(stream, fileEncoding));
//            input.setSystemId(XMLRegister.getProperty("xml.dtd.dir"));
            
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            
            // Set validation status.
            dbf.setValidating(parseValidation);
            
            
            // Get DocumentBuilder instance and analyze xml file.
            DocumentBuilder parser = dbf.newDocumentBuilder();
            
            Document document = parser.parse(input);

            return document;
            
        }
        catch (Exception e) {
            throw new XMLException("Can't read xml file.", e);
        }
    }
    
    /**
     * Read argument's file as XML document based on DOM.
     * Returns org.w3c.dom.Document object that be reading result.
     * 
     * @param xmlFile   java.io.File as xml file.
     * @return org.w3c.dom.Document Object.
     * @throws XMLException throw when reader failed to read files.
     */
    public Document readAsDocument(File xmlFile) throws XMLException {

        try {
            if (xmlFile.isDirectory()) {
                throw new XMLException(
                        xmlFile + " is directory. must be a file.");
            }
            
            // returns target that is xml document.
            return this.readAsDocument(new FileInputStream(xmlFile));
        }
        catch (Exception e) {
            throw new XMLException(
                    xmlFile.getPath() + " can't read.", e);
        }
    }
    
    /**
     * Set parser's dtd validation process enable.
     * @param validation    set 'true' if do validation.
     */
    public void setValidation(boolean validation) {
        this.parseValidation = validation;
    }
}
