/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.commons/src/jp/sourceforge/jeextension/common/xml/XMLNodeBuilder.java,v 1.1 2004/12/17 02:25:24 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2004/12/17 02:25:24 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.common.xml;

import java.util.Map;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;


/**
 * XMLNodeBuilder is that build XMLNode form org.w3c.dom.Document object.
 * it transform org.w3c.Document's node structure to XMLNode's node structure.
 * deferences of XMLNode and org.w3c.dom.Document, see
 * {@link jp.sourceforge.jeextension.common.xml.XMLNode}
 */
public final class XMLNodeBuilder {

    /**
     * org.w3c.dom.Document as source.
     */
    private Document sourceDocument;

    /**
     * Initiate this object using org.w3c.dom.Document as source.
     * @param doc   org.w3c.dom.Document object.
     */
    XMLNodeBuilder(Document doc) {
        super();
        this.sourceDocument = doc;
    }

    /**
     * Build XMLNode from org.w3c.dom.Document object as source.
     * @return created XMLNode Object.
     * @throws XMLException throws when builder failed to transform one.
     */
    public XMLNode build() throws XMLException {
        XMLNode docXMLNode = this.createXMLNode();
        Element docElement = sourceDocument.getDocumentElement();

        buildXMLNode(docElement.getFirstChild(), docXMLNode);
        
        return docXMLNode;
    }
    
    /**
     * Transform to XMLNode form DocumentElement of org.w3c.dom.Element.
     * This process do transform org.w3c.dom.Element to XMLNode only. don't
     * transform org.w3c.dom.Element of one's child element.
     *
     * @return transformed XMLNode object.
     */
    private XMLNode createXMLNode() {
        // Get DocumentElement object from Document object.
        Element docElement = sourceDocument.getDocumentElement();

        // Set name to XMLNode object.
        XMLNode docXMLNode = new XMLNode(docElement.getNodeName());

        // Set attributes of DocumentElement to XMLNode
        if (docElement.hasAttributes()) {
            this.buildAttrs(docElement, docXMLNode);
        }
        return docXMLNode;
    }

    /**
     * transform all child's org.w3c.dom.Node to XMLNode reflexively.
     * transformation type see {@link XMLNode}
     *
     * @param node      target's org.w3c.dom.Node
     * @param xmlNode   parent target XMLNode
     * @throws XMLException throws when set child to XMLNode has value.
     */
    private void buildXMLNode(Node node, XMLNode xmlNode)
            throws XMLException {

        for (; node != null; node = node.getNextSibling()) {
            // Process ELEMENT_NODE only.
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                // Create new XMLNode.
                XMLNode childXMLNode = new XMLNode(node.getNodeName());
                // Set attributes
                if (node.hasAttributes()) {
                    buildAttrs(node, childXMLNode);
                }
                // Set value.
                this.buildNodeValue(node.getFirstChild(), childXMLNode);
                // Set XMLNode to parent.
                xmlNode.addChild(childXMLNode);
                // reflexive call.
                this.buildXMLNode(node.getFirstChild(), childXMLNode);
            }
        }
    }

    /**
     * Copy org.w3c.dom.NamedNodeMap's values to java.util.Map using attribute
     * name key's.<br>
     * Copy target must be String types charactor.<br>
     * Copied Key and Value be String types object.
     *
     * @param node      transform target's parent org.w3c.dom.Node
     * @param xmlNode   transform target's parent XMLNode
     */
    private void buildAttrs(Node node, XMLNode xmlNode) {
        NamedNodeMap attrNodeMap = node.getAttributes();
        Map attrMap = xmlNode.getAttributes();
        for (int i = 0; i < attrNodeMap.getLength(); i++) {
            Attr attr = (Attr) attrNodeMap.item(i);

            // Set attribute name and value to Map.
            String attrName  = attr.getName();
            String attrValue = attr.getValue();
            attrMap.put(attrName, attrValue);
        }
    }

    /**
     * Set value to XMLNode form org.w3c.dom.Node.
     *
     * @param node      source of org.w3c.dom.Node having value.
     * @param xmlNode   distination of XMLNode set to.
     */
    private void buildNodeValue(Node node, XMLNode xmlNode) {
        
        for (; node != null; node = node.getNextSibling()) {

            StringBuffer sb = new StringBuffer();
            for (; node != null; node = node.getNextSibling()) {
                sb.append(node.getNodeValue());
            }
            String nodeValue = sb.toString();
            if (!nodeValue.startsWith("\n")) {
                xmlNode.setNodeValue(nodeValue);
            }
            break;
        }
    }
}
