/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.commons/src/jp/sourceforge/jeextension/common/xml/XMLNode.java,v 1.1 2004/12/17 02:25:24 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2004/12/17 02:25:24 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.common.xml;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * XMLNode is class that managing same org.w3c.dom.Node.
 * <br>
 * but don't have Node.TEXT_NODE deferences org.w3c.dom.Node.
 * Node.TEXT_NODE's value has most under child node only.
 * <br><br>
 * ex)DXMLNode structure as org.w3c.dom.Node
 * <pre>
 * &lt;documentElement&gt;
 *     &lt;element1&gt;
 *         &lt;element2&gt;
 *             &lt;element3&gt;value&lt;/element3&gt;
 *         &lt;/element2&gt;
 *     &lt;/element&gt;
 *     &lt;element1&gt;
 *         &lt;element2&gt;
 *            &lt;element3&gt;value&lt;/element3&gt;
 *         &lt;/element2&gt;
 *     &lt;/element1&gt;
 * &lt;/documentElement&gt;
 *
 * Node(Node.DOCUMENT_NODE, "documentElement")
 *   Node(Node.TEXT_NODE, "#text")
 *   Node(Node.ELEMENT_NODE, "element1")
 *     Node(Node.TEXT_NODE, #text)
 *     Node(Node.ELEMENT_NODE, "element2")
 *       Node(Node.TEXT_NODE, #text)
 *       Node(Node.ELEMENT_NODE, "element3")
 *         Node(Node.TEXT_NODE, #text)
 *         Node(Node.TEXT_NODE, "value")
 *         Node(Node.TEXT_NODE, #text)
 *       Node(Node.TEXT_NODE, #text)
 *     Node(Node.TEXT_NODE, #text)
 *   Node(Node.TEXT_NODE, #text)
 *   Node(Node.ELEMENT_NODE, "element1")
 *     Node(Node.TEXT_NODE, #text)
 *     Node(Node.ELEMENT_NODE, "element2")
 *       Node(Node.TEXT_NODE, #text)
 *       Node(Node.ELEMENT_NODE, "element3")
 *         Node(Node.TEXT_NODE, #text)
 *         Node(Node.TEXT_NODE, "value")
 *         Node(Node.TEXT_NODE, #text)
 *       Node(Node.TEXT_NODE, #text)
 *     Node(Node.TEXT_NODE, #text)
 *   Node(Node.TEXT_NODE, #text)
 *
 * XMLNode("documentElement")
 *   XMLNode("element1")
 *     XMLNode("element2")
 *       XMLNode("element3", "value")
 *   XMLNode("element1")
 *     XMLNode("element2")
 *       XMLNode("element3", "value")
 * </pre>
 */
public final class XMLNode {

    /**
     * attribute map for this node.
     */
    private Map attributes = new LinkedHashMap();

    /**
     * child's node list. it's empty if don't have child.
     */
    private XMLNodeList childNodeList = new XMLNodeList();

    /**
     * this node name.
     */
    private String nodeName;

    /**
     * this node value. it's "" if have child.
     */
    private String nodeValue;

    /**
     * this parent node. it's null if don't have parent or this is root.
     */
    private XMLNode parentNode;

    /**
     * Initiate this object using name of this node.
     * <br>
     * set "" to node value as default.
     * <br><br>
     * @param name  name of node.
     */
    public XMLNode(String name) {
        this(name, "");
    }

    /**
     * Initiate this object using name and value of this node.
     * <br><br>
     * @param name  name of node.
     * @param value value of node.
     */
    public XMLNode(String name, String value) {
        super();
        this.nodeName = name;
        this.nodeValue = value;
    }
    
    /**
     * Add child node to this node.
     * @param add   child node for adding.
     * @throws IllegalArgumentException throws when this node have node value.
     *                                  Only parent node that don't have node
     *                                  value, enable add child.
     */
    public void addChild(XMLNode add) {
        if (!nodeValue.equals("")) {
            throw new IllegalArgumentException(
                "can't add child node that having value[" + nodeValue + "].");
        }
        this.childNodeList.add(add);
        add.setParent(this);
    }

    /**
     * Returns attribute of paremeter key.<br>
     * if can't find target attribute, return null.<br>
     * <br>
     * @param name name of attribute(key).
     * @return attribute as String.
     */
    public String getAttribute(String name) {
        return (String) this.attributes.get(name);
    }

    /**
     * Returns this arrtibute manaing map object.
     * <br><br>
     * @return Map having all attributes entry.
     */
    public Map getAttributes() {
        return this.attributes;
    }

    /**
     * Return this child node list.
     * if list size is 0, return empty list.
     * <br><br>
     * @return child node list.
     */
    public XMLNodeList getChildNodes() {
        return this.childNodeList;
    }

    /**
     * Returns name of this node.
     * <br><br>
     * @return name of this node.
     */
    public String getNodeName() {
        return this.nodeName;
    }

    /**
     * Returns value of this node.
     * if have child node or be default state, return "".
     * <br><br>
     * @return value of this node.
     */
    public String getNodeValue() {
        return this.nodeValue;
    }

    /**
     * Return parent of this node.
     * if don't have parent, return null.
     * <br><br>
     * @return parent node.
     */
    public XMLNode getParentNode() {
        return this.parentNode;
    }

    /**
     * Exist attribute of target's key.
     * <br><br>
     * @param name attribute to the key.
     * @return return true that have any attribute, else return false.
     */
    public boolean hasAttribute(String name) {
        return this.attributes.containsKey(name);
    }

    /**
     * Exist attributes.
     * <br><br>
     * @return if attributes exists, return true. else return false.
     */
    public boolean hasAttributes() {
        return !this.attributes.isEmpty();
    }

    /**
     * Exist child nodes.
     * <br><br>
     * @return if child nodes exists, return true. else return false.
     */
    public boolean hasChildNodes() {
        return !this.childNodeList.isEmpty();
    }

    /**
     * Remove attribute from this attribute managing map.
     * <br><br>
     * @param name  key of attributes.
     */
    public void removeAttribute(String name) {
        this.attributes.remove(name);
    }

    /**
     * Remove child node from this child node list.
     * <br><br>
     * @param node  node that deleting.
     */
    public void removeChild(XMLNode node) {
        this.childNodeList.remove(node);
    }

    /**
     * Set attribute using name and value.
     * <br><br>
     * @param name  name of attribute.
     * @param value value of attribute.
     */
    public void setAttribute(String name, String value) {
        this.attributes.put(name, value);
    }

    /**
     * Set node value to this node.
     * <br><br>
     * @param value value of node.
     */
    public void setNodeValue(String value) {
        this.nodeValue = value;
    }

    /**
     * Set parent to this node.
     * <br><br>
     * @param parent    node that depends one.
     */
    private void setParent(XMLNode parent) {
        this.parentNode = parent;
    }

    /**
     * Return String values that meaning String.
     * <br><br>
     * @return node state as String.
     */
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        buffer.append("[nodeName=");
        buffer.append(this.nodeName);
        buffer.append(", nodeValue=");
        buffer.append(this.nodeValue);
        buffer.append(", chileNodes=");
        buffer.append(this.childNodeList.size());

        if (this.childNodeList.size() != 0) {
            buffer.append("[");
            buffer.append(this.childNodeList.toString());
            buffer.append("]");
        }

        buffer.append(", attributes=");
        buffer.append(this.attributes.size());

        if (this.attributes.size() != 0) {
            buffer.append("[");
            buffer.append(this.attributes.toString());
            buffer.append("]");
        }

        buffer.append("]");
        return buffer.toString();
    }
    
    /**
     * Compare this node and target of parameter.
     * comparation process is ...
     * <br>
     * 1. Types<br>
     * 2. name of node<br>
     * 3. value of node<br>
     * 4. attributes of node<br>
     * 5. child node size.<br>
     * 6. check chidl node state. 1 to 5<br>
     * <br>
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object obj) {
        // 1. is euqal Class Types.
        if (obj.getClass() != getClass()) {
            return false;
        }
        // 2. check node name.
        XMLNode target = (XMLNode) obj;
        if (!target.getNodeName().equals(nodeName)) {
            return false;
        }
        // 3. check node value.
        if (!target.getNodeValue().equals(nodeValue)) {
            return false;
        }
        // 4. check node attributes.
        if (!target.getAttributes().equals(attributes)) {
            return false;
        }
        
        // 5. check child node size.
        List targetChildList = target.getChildNodes();
        List thisChildList   = getChildNodes();
        if (targetChildList.size() != thisChildList.size()) {
            return false;
        }
        
        // 6. check all child node each other.
        for (int i = 0; i < targetChildList.size(); i++) {
            XMLNode targetChild = (XMLNode) targetChildList.get(i);
            XMLNode thisChild   = (XMLNode) thisChildList.get(i);
            boolean result = thisChild.equals(targetChild);
            if (!result) {
                return false;
            }
        }
        // if all equals 1 to 6, return true
        return true;
    }
    
    /**
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return super.hashCode();
    }

}
