/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.commons/src/jp/sourceforge/jeextension/common/xml/XMLFile.java,v 1.1 2004/12/17 02:25:24 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2004/12/17 02:25:24 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.common.xml;

import java.io.File;

import org.w3c.dom.Document;

/**
 * XMLFile is java.io.File wrapper object that means xml file.<br>
 * This class have XMLWriter and XMLReader interface.
 */
public class XMLFile {

    /** wrapped java.io.File object. */
    protected File currentFile;
    
    /** current file encoding. */
    protected String fileEncoding;
    
    /**
     * Initiate this object using file path of target.
     * @param filePath  file path as String.
     */
    public XMLFile(String filePath) {
        this(filePath, System.getProperty("file.encoding"));
    }
    
    /**
     * Initiate this object using file path of target.
     * @param filePath  file path as String.
     * @param encoding  file encoding.
     */
    public XMLFile(String filePath, String encoding) {
        this(new File(filePath.replace('/', File.separatorChar)), encoding);
    }
    
    /**
     * Initiate this object using java.io.File Object of target.
     * @param file  file as java.io.File.
     */
    public XMLFile(File file) {
        this(file, System.getProperty("file.encoding"));
    }
    
    /**
     * Initiate this object using java.io.File Object of target.
     * @param file      file as java.io.File.
     * @param encoding  file encoding.
     */
    public XMLFile(File file, String encoding) {
        this.fileEncoding = encoding;
        this.currentFile = file;
        if (this.currentFile.isDirectory() == true) {
            throw new IllegalArgumentException(file.getPath()
                + " isn't File object."
                + " can't initiate except file path.");
        }
    }

    /**
     * Returns wrapped java.io.File object.
     *
     * @return java.io.File object that wrapped.
     */
    public File getFile() {
        return this.currentFile;
    }
    
    /**
     * Write org.w3c.dom.Document to wrapped File object.
     * 
     * @param document  Node of ord.w3c.dom.Document.
     * @throws XMLException throws when writer failed to write one.
     */
    public void writeDocument(Document document) throws XMLException {
        XMLWriter writer = new XMLWriter(fileEncoding);
        writer.write(this.currentFile, document);
    }

    /**
     * Write XMLNode to wrapped File object.
     *
     * @param xmlNode   Node of XMLNode.
     * @throws XMLException throws when writer failed to write one, or current
     *                      file path isn't file or be directory.
     */
    public void writeNode(XMLNode xmlNode) throws XMLException {
        XMLWriter writer = new XMLWriter(fileEncoding);
        writer.write(this.currentFile, xmlNode);
    }

    /**
     * Read XMLNode object from current xml file path.
     * and returns XMLNode object that read from file.
     *
     * @return XMLNode object meaning target's xml file nodes.
     * @throws XMLException throws when reader failed to read one, or current
     *                      file path isn't file or be durectory.
     */
    public XMLNode readNode() throws XMLException {
        XMLReader reader = new XMLReader(fileEncoding);
        return reader.readAsXMLNode(this.currentFile);
    }
    
    /**
     * Read org.w3c.dom.Document form current cml file path.
     * and returns org.w3c.dom.Document.
     * 
     * @return org.w3c.dom.Document meaing target's xml file nodes.
     * @throws XMLException throws when reader failed to read one, or current
     *                      file path isn't file or be durectory.
     */
    public Document readDocument() throws XMLException {
        XMLReader reader = new XMLReader(fileEncoding);
        return reader.readAsDocument(this.currentFile);
    }

}
