/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.commons/src/jp/sourceforge/jeextension/common/xml/XMLDocumentBuilder.java,v 1.1 2004/12/17 02:25:24 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2004/12/17 02:25:24 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.common.xml;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.Text;


/**
 * XMLDocumentBuilder is that build org.w3c.dom.Document from XMLNode object.
 * it transform XMLNode's node structure to org.w3c.dom.Document's node
 * structure. deferences of XMLNode and org.w3c.dom.Document, see
 * {@link jp.sourceforge.jeextension.common.xml.XMLNode}.
 */
public final class XMLDocumentBuilder {

    /** for reference of Element and Text object when build one. */
    private Document document;

    /** target XMLNode for building from org.w3c.dom.Document. */
    private XMLNode sourceXMLNode;

    /**
     * Initiate this object using source of XMLNode.
     * @param node XMLNode object.
     */
    XMLDocumentBuilder(XMLNode node) {
        super();
        this.sourceXMLNode = node;
    }

    /**
     * Build org.w3c.dom.Document from source.
     * @return build org.w3c.dom.Document object.
     * @throws XMLException throws when parser failed to create
     *                      org.w3c.dom.Document.
     */
    public Document build() throws XMLException {

        // Create root element.
        Element rootElement = this.createDocumentElement();

        // Build DocumentElement from XMLNode.
        if (sourceXMLNode.hasChildNodes()) {
            this.buildDocument(rootElement, sourceXMLNode);
        }
        else {
            // Set value if no more element.
            String nodeValue = sourceXMLNode.getNodeValue();
            Text value = this.document.createTextNode(nodeValue);
            // Set attribute if no more element.
            if (sourceXMLNode.hasAttributes()) {
                copyAttrs(rootElement, sourceXMLNode);
            }
            rootElement.appendChild(value);
        }

        // Set child DocumentElement to Document. complete building.
        document.appendChild(rootElement);
        return document;
    }

    /**
     * Transform XMLNode to org.w3c.dom.Node reflexively.
     * transform types see {@link XMLNode}.
     *
     * @param parentNode    target parent org.w3c.dom.Node object.
     * @param parentXMLNode target parent XMLNode object.
     */
    private void buildDocument(Node parentNode, XMLNode parentXMLNode) {
        List childXMLNodeList = parentXMLNode.getChildNodes();

        for (int i = 0; i < childXMLNodeList.size(); i++) {

            // Get child node of XMLNode
            XMLNode childXMLNode = (XMLNode) childXMLNodeList.get(i);

            // transform XMLNode to Element.
            Element childNode = this.toElement(childXMLNode);

            // process reflexively if one has child node.
            if (childXMLNode.hasChildNodes()) {
                buildDocument(childNode, childXMLNode);
            }
            parentNode.appendChild(childNode);
        }
    }

    /**
     * Copy attribute name and values form org.w3c.dom.NamedNodeMap to
     * java.util.Map object.<br>
     * Copy target must be String types charactor.<br>
     * Copied Key and Value be String types object.
     *
     * @param node      transform target's parent org.w3c.dom.Node
     * @param xmlNode   transform target's parent XMLNode
     */
    private void copyAttrs(Node node, XMLNode xmlNode) {
        Map      map      = xmlNode.getAttributes();
        Iterator iterator = map.keySet().iterator();

        while (iterator.hasNext()) {
            String attrName  = (String) iterator.next();
            String attrValue = (String) map.get(attrName);
            Attr   attr      = this.document.createAttribute(attrName);
            attr.setValue(attrValue);
            NamedNodeMap attrNodeMap = node.getAttributes();
            attrNodeMap.setNamedItem(attr);
        }
    }

    /**
     * Transform to org.w3c.dom.Element form XMLNode.
     * This process do transform XMLNode to org.w3c.dom.Element only. don't
     * transform XMLNode of one's child element.
     *
     * @return transformed org.w3c.dom.Element object.
     * @throws XMLException throws when parser failed to create
     *                      org.w3c.dom.Document
     */
    private Element createDocumentElement() throws XMLException {

        try {
            // create document using JAXP API.
            DocumentBuilderFactory factory    =
                DocumentBuilderFactory.newInstance();
            DocumentBuilder        builder = factory.newDocumentBuilder();
            this.document = builder.newDocument();

            // create DocumentElement from XMLNode.
            Element docElement =
                this.document.createElement(sourceXMLNode.getNodeName());

            // set attributes to DocumentElement from XMLNode.
            if (sourceXMLNode.hasAttributes()) {
                this.copyAttrs(docElement, sourceXMLNode);
            }
            return docElement;
        }
        catch (ParserConfigurationException e) {
            throw new XMLException(e);
        }
    }

    /**
     * Transform XMLNode to org.w3c.dom.Node object.
     * copy targets is node name and value of target Node.
     *
     * @param targetXMLNode target XMLNode.
     * @return created Element.
     */
    private Element toElement(XMLNode targetXMLNode) {

        // get name and value of XMLNode.
        String childNodeName  = targetXMLNode.getNodeName();
        String childNodeValue = targetXMLNode.getNodeValue();

        // create Element from name of node.
        Element element = this.document.createElement(childNodeName);

        if (!childNodeValue.equals("")) {
            Text value = this.document.createTextNode(childNodeValue);
            element.appendChild(value);
        }

        // Copy attributes to Element from XMLNode.
        if (targetXMLNode.hasAttributes()) {
            copyAttrs(element, targetXMLNode);
        }
        return element;
    }
}
