/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IConvolutionSupport;
import ch.kuramo.javie.api.services.IVideoEffectContext;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Sharpen", category=Categories.BLUR_AND_SHARPEN)
public class Sharpen {

	@Property(value="0", min="0")
	private IAnimatableDouble sharpenAmount;


	private final IVideoEffectContext context;

	private final IConvolutionSupport convolution;

	@Inject
	public Sharpen(IVideoEffectContext context, IConvolutionSupport convolution) {
		this.context = context;
		this.convolution = convolution;
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		if (source.getBounds().isEmpty()) {
			return source;
		}

		double sharpenAmount = context.value(this.sharpenAmount) / 20;		// 20で割るとAEと同じくらいになる。
		if (sharpenAmount == 0) {
			return source;
		}

		float a = (float)(-sharpenAmount/9);
		float b = (float)(1+8*sharpenAmount/9);
		float[] kernel = { a, a, a, a, b, a, a, a, a };

		try {
			source.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
			return convolution.convolve2D(source, null, 3, kernel);
		} finally {
			source.dispose();
		}
	}

}
