/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Collection;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;
import javax.media.opengl.glu.GLU;
import javax.media.opengl.glu.GLUtessellator;
import javax.media.opengl.glu.GLUtessellatorCallback;
import javax.media.opengl.glu.GLUtessellatorCallbackAdapter;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.internal.VideoBufferImpl;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class VideoRenderSupportImpl implements IVideoRenderSupport {

	private static final Logger _logger = LoggerFactory.getLogger(VideoRenderSupportImpl.class);


	private final RenderContext context;

	private final IArrayPools arrayPools;

	private final GLGlobal glGlobal;

	@Inject
	public VideoRenderSupportImpl(RenderContext context, IArrayPools arrayPools, GLGlobal glGlobal) {
		this.context = context;
		this.arrayPools = arrayPools;
		this.glGlobal = glGlobal;
	}

	public IVideoBuffer createVideoBuffer(VideoBounds bounds) {
		return createVideoBuffer(bounds, context.getColorMode());
	}

	public IVideoBuffer createVideoBuffer(VideoBounds bounds, ColorMode colorMode) {
		return new VideoBufferImpl(colorMode, bounds, context, this, arrayPools, glGlobal);
	}

	private int limitMaxTextureImageUnits(int n) {
		int max = glGlobal.getMaxTextureImageUnits();
		if (n > max) {
			_logger.warn(String.format(
					"number of texture (%d) exceeds GL_MAX_TEXTURE_IMAGE_UNITS (%d)",
					n, max));

			n = max;
		}
		return n;
	}

	private void useFramebufferMRT(
			Runnable operation, int pushAttribs, boolean enableTexture,
			IVideoBuffer[] output, IVideoBuffer... input) {

		GL2 gl = context.getGL().getGL2();

		int[] fb = new int[1];
		gl.glGetFramebufferAttachmentParameteriv(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, fb, 0);
		if (fb[0] != 0) {
			throw new IllegalStateException("framebuffer is in use");
		}

		// glDrawBuffer は毎回設定することが前提なので GL_COLOR_BUFFER_BIT は push しなくても問題無いだろう。
		// GL_TEXTURE_2D は GL_TEXTURE_BIT で保存されるはずだが、一部の環境では保存されないっぽい。
		// TODO ATI で GL2.GL_ALL_ATTRIB_BITS を指定されるとエラーになるので、エラーになるビットを下げる。

		gl.glPushAttrib(pushAttribs
			//	| GL2.GL_COLOR_BUFFER_BIT					// glDrawBuffer  [GL_DRAW_BUFFER]
			//	| GL2.GL_CURRENT_BIT						// glColor4f     [GL_CURRENT_COLOR]
				| GL2.GL_TEXTURE_BIT						// glBindTexture [GL_TEXTURE_BINDING_2D]
				| (enableTexture ? GL2.GL_ENABLE_BIT : 0));	// glEnable      [GL_TEXTURE_2D]
		try {

			// TODO MRTの使用可能な数をチェックする必要がある。

			int[] drawBuffers = new int[output.length];
			for (int i = 0; i < output.length; ++i) {
				drawBuffers[i] = GL2.GL_COLOR_ATTACHMENT0 + i;
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						drawBuffers[i], GL2.GL_TEXTURE_2D, output[i].getTexture(), 0);
			}
			gl.glDrawBuffers(drawBuffers.length, drawBuffers, 0);

			for (int i = 0, n = limitMaxTextureImageUnits(input.length); i < n; ++i) {
				gl.glActiveTexture(GL2.GL_TEXTURE0 + i);
				gl.glBindTexture(GL2.GL_TEXTURE_2D, input[i].getTexture());
				if (enableTexture) {
					gl.glEnable(GL2.GL_TEXTURE_2D);
				}
			}

			//gl.glColor4f(1, 1, 1, 1);

			operation.run();

		} finally {
			for (int i = 0; i < output.length; ++i) {
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0+i, GL2.GL_TEXTURE_2D, 0, 0);
			}

			gl.glPopAttrib();
		}
	}

	private void useFramebuffer(
			Runnable operation, int pushAttribs, boolean enableTexture,
			IVideoBuffer output, IVideoBuffer... input) {

		useFramebufferMRT(operation, pushAttribs, enableTexture, new IVideoBuffer[] { output }, input);
	}

	public IVideoBuffer useFramebuffer(
			Runnable operation, int pushAttribs,
			IVideoBuffer output, IVideoBuffer... input) {

		IVideoBuffer buffer = null;
		if (output == null) {
			if (input.length == 0) {
				throw new IllegalArgumentException("when output is null, at least one input must be specified.");
			}
			buffer = output = createVideoBuffer(input[0].getBounds());
		}

		try {
			useFramebuffer(operation, pushAttribs, true, output, input);

			buffer = null;
			return output;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	public IVideoBuffer useShaderProgram(
			IShaderProgram program, Collection<GLUniformData> uniforms,
			Runnable operation, int pushAttribs,
			IVideoBuffer output, IVideoBuffer... input) {

		IVideoBuffer buffer = null;
		if (output == null) {
			if (input.length == 0) {
				throw new IllegalArgumentException("when output is null, at least one input must be specified.");
			}
			buffer = output = createVideoBuffer(input[0].getBounds());
		}

		try {
			useShaderProgramMRT(program, uniforms, operation, pushAttribs, new IVideoBuffer[] { output }, input);

			buffer = null;
			return output;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	public void useShaderProgramMRT(
			final IShaderProgram program, final Collection<GLUniformData> uniforms,
			final Runnable operation, int pushAttribs,
			IVideoBuffer[] output, IVideoBuffer... input) {

		Runnable useProgram = new Runnable() {
			public void run() {
				program.useProgram(new Runnable() {
					public void run() {
						GL2 gl = context.getGL().getGL2();
						for (GLUniformData data : uniforms) {
							data.setLocation(program.getUniformLocation(data.getName()));
							gl.glUniform(data);
						}
						operation.run();
					}
				});
			}
		};

		useFramebufferMRT(useProgram, pushAttribs, false, output, input);
	}

	public IVideoBuffer useShaderProgram(
			IShaderProgram program, Collection<GLUniformData> uniforms,
			IVideoBuffer output, IVideoBuffer... input) {

		IVideoBuffer buffer = null;
		if (output == null) {
			if (input.length == 0) {
				throw new IllegalArgumentException("when output is null, at least one input must be specified.");
			}
			buffer = output = createVideoBuffer(input[0].getBounds());
		}

		try {
			useShaderProgramMRT(program, uniforms, new IVideoBuffer[] { output }, input);

			buffer = null;
			return output;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	public void useShaderProgramMRT(
			IShaderProgram program, Collection<GLUniformData> uniforms,
			final IVideoBuffer[] output, final IVideoBuffer... input) {

		Runnable operation = new Runnable() {
			public void run() {
				ortho2D(output[0]);
				quad2D(output[0], input);
			}
		};

		useShaderProgramMRT(program, uniforms, operation, 0, output, input);
	}

	private void ortho2D(double left, double top, int width, int height) {
		GL2 gl = context.getGL().getGL2();
		GLU glu = context.getGLU();

		gl.glViewport(0, 0, width, height);

		gl.glMatrixMode(GL2.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluOrtho2D(left, left+width, top, top+height);

		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glLoadIdentity();
	}

	public void ortho2D(IVideoBuffer output) {
		ortho2D(output.getBounds());
	}

	public void ortho2D(VideoBounds bounds) {
		ortho2D(bounds.x, bounds.y, bounds.width, bounds.height);
	}

	private double[][][] toTexCoords(IVideoBuffer output, IVideoBuffer... input) {
		VideoBounds outputBounds = output.getBounds();
		double x = outputBounds.x;
		double y = outputBounds.y;
		int w = outputBounds.width;
		int h = outputBounds.height;

		int n = input.length;
		double[][][] texCoords = new double[n][][];

		for (int i = 0; i < n; ++i) {
			VideoBounds inputBounds = input[i].getBounds();
			double dx = x - inputBounds.x;
			double dy = y - inputBounds.y;
			double iw = inputBounds.width;
			double ih = inputBounds.height;
			texCoords[i] = new double[][] { {dx/iw, dy/ih}, {(dx+w)/iw, dy/ih}, {(dx+w)/iw, (dy+h)/ih}, {dx/iw, (dy+h)/ih} };
		}

		return texCoords;
	}

	public void quad2D(IVideoBuffer output, IVideoBuffer... input) {
		quad2D(output.getBounds(), toTexCoords(output, input));
	}

	public void quad2D(VideoBounds bounds, double[][]... texCoords) {
		double left = bounds.x;
		double top = bounds.y;
		double right = left + bounds.width;
		double bottom = top + bounds.height;

		quad2D(left, top, right, bottom, texCoords);
	}

	public void quad2D(double left, double top, double right, double bottom, double[][]... texCoords) {
		int n = texCoords.length;

		GL2 gl = context.getGL().getGL2();
		gl.glBegin(GL2.GL_QUADS);

		for (int i = 0; i < n; ++i) {
			gl.glMultiTexCoord2f(GL2.GL_TEXTURE0+i, (float)texCoords[i][0][0], (float)texCoords[i][0][1]);
		}
		gl.glVertex2f((float)left, (float)top);

		for (int i = 0; i < n; ++i) {
			gl.glMultiTexCoord2f(GL2.GL_TEXTURE0+i, (float)texCoords[i][1][0], (float)texCoords[i][1][1]);
		}
		gl.glVertex2f((float)right, (float)top);

		for (int i = 0; i < n; ++i) {
			gl.glMultiTexCoord2f(GL2.GL_TEXTURE0+i, (float)texCoords[i][2][0], (float)texCoords[i][2][1]);
		}
		gl.glVertex2f((float)right, (float)bottom);

		for (int i = 0; i < n; ++i) {
			gl.glMultiTexCoord2f(GL2.GL_TEXTURE0+i, (float)texCoords[i][3][0], (float)texCoords[i][3][1]);
		}
		gl.glVertex2f((float)left, (float)bottom);

		gl.glEnd();
	}

	public void polygon2D(double[][][] contours, IVideoBuffer output, IVideoBuffer... input) {
		polygon2D(contours, output.getBounds(), toTexCoords(output, input));
	}

	public void polygon2D(double[][][] contours, final VideoBounds bounds, final double[][]... texCoords) {

		GLUtessellatorCallback callback = new GLUtessellatorCallbackAdapter() {
			final GL2 gl = context.getGL().getGL2();

			@Override
			public void begin(int type) {
				gl.glBegin(type);
			}

			@Override
			public void end() {
				gl.glEnd();
			}

			@Override
			public void vertex(Object vertexData) {
				double[] vertex = (double[]) vertexData;

				double px = (vertex[0]-bounds.x)/bounds.width;
				double py = (vertex[1]-bounds.y)/bounds.height;

				for (int i = 0, n = texCoords.length; i < n; ++i) {
					double x1 = texCoords[i][0][0]*(1-px) + texCoords[i][1][0]*px;
					double x2 = texCoords[i][3][0]*(1-px) + texCoords[i][2][0]*px;
					double x = x1*(1-py) + x2*py;

					double y1 = texCoords[i][0][1]*(1-py) + texCoords[i][3][1]*py;
					double y2 = texCoords[i][1][1]*(1-py) + texCoords[i][2][1]*py;
					double y = y1*(1-px) + y2*px;

					gl.glMultiTexCoord2f(GL2.GL_TEXTURE0+i, (float)x, (float)y);
				}

				gl.glVertex2f((float)vertex[0], (float)vertex[1]);
			}

			@Override
			public void combine(double[] coords, Object[] data, float[] weight, Object[] outData) {
				outData[0] = new double[] { coords[0], coords[1], coords[2] };
			}

			@Override
			public void error(int errnum) {
				System.err.println("Tessellation Error: " + context.getGLU().gluErrorString(errnum));
			}
		};

		GLUtessellator tess = GLU.gluNewTess();
		try {
			GLU.gluTessCallback(tess, GLU.GLU_TESS_BEGIN, callback);
			GLU.gluTessCallback(tess, GLU.GLU_TESS_END, callback);
			GLU.gluTessCallback(tess, GLU.GLU_TESS_VERTEX, callback);
			GLU.gluTessCallback(tess, GLU.GLU_TESS_COMBINE, callback);
			GLU.gluTessCallback(tess, GLU.GLU_TESS_ERROR, callback);

			GLU.gluTessBeginPolygon(tess, null);
			for (double[][] points : contours) {
				GLU.gluTessBeginContour(tess);
				for (double[] pt : points) {
					double[] vertex = new double[] { pt[0], pt[1], 0 };
					GLU.gluTessVertex(tess, vertex, 0, vertex);
				}
				GLU.gluTessEndContour(tess);
			}
			GLU.gluTessEndPolygon(tess);

		} finally {
			GLU.gluDeleteTess(tess);
		}
	}

	public void copy(IVideoBuffer src, IVideoBuffer dst) {
		final VideoBounds srcBounds = src.getBounds();
		final VideoBounds dstBounds = dst.getBounds();

		// TODO 重なっている領域が無い場合も何もしなくてよい。
		if (srcBounds.isEmpty() || dstBounds.isEmpty()) {
			return;
		}

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();
				gl.glColor4f(1, 1, 1, 1);

				ortho2D(dstBounds);
				quad2D(srcBounds, new double[][] { { 0, 0 }, { 1, 0 }, { 1, 1 }, { 0, 1 } });
			}
		};
		int pushAttribs = GL2.GL_CURRENT_BIT;
		useFramebuffer(operation, pushAttribs, dst, src);
	}

}
