/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.lang.ref.PhantomReference;
import java.lang.ref.ReferenceQueue;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;

import javax.media.opengl.GL;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.services.IVBOCache;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class VBOCacheImpl implements IVBOCache {

	private static final Logger logger = LoggerFactory.getLogger(VBOCacheImpl.class);


	private final ReferenceQueue<Object> refq = new ReferenceQueue<Object>();

	private final WeakHashMap<Object, VBOCacheRecord> cache = new WeakHashMap<Object, VBOCacheRecord>();

	private final Map<VBOCacheRecord, VBOCacheReference> refmap = new HashMap<VBOCacheRecord, VBOCacheReference>(); 


	private final RenderContext context;

	@Inject
	public VBOCacheImpl(RenderContext context) {
		this.context = context;
	}

	void dispose() {
		for (VBOCacheReference ref : refmap.values()) {
			ref.enqueue();
		}
		pollAndDelete();
	}

	void pollAndDelete() {
		GL gl = context.getGL();
		int[] vboId = new int[1];

		VBOCacheReference ref;
		while ((ref = (VBOCacheReference) refq.poll()) != null) {
			if (ref.rec != null) {
				refmap.remove(ref.rec);
				vboId[0] = ref.rec.vboId;
				gl.glDeleteBuffers(1, vboId, 0);

				logger.info("deleted cached vbo");
			}
		}
	}

	public Object put(Object referent, VBOCacheRecord rec) {
		Object oldData = delete(referent);

		cache.put(referent, rec);
		refmap.put(rec, new VBOCacheReference(referent, rec, refq));

		return oldData;
	}

	public VBOCacheRecord get(Object referent) {
		return cache.get(referent);
	}

	public VBOCacheRecord remove(Object referent) {
		VBOCacheRecord rec = cache.remove(referent);
		VBOCacheReference ref = refmap.remove(rec);
		if (ref != null) {
			ref.rec = null;
		}
		return rec;
	}

	public Object delete(Object referent) {
		VBOCacheRecord rec = remove(referent);
		if (rec != null) {
			GL gl = context.getGL();
			gl.glDeleteBuffers(1, new int[] { rec.vboId }, 0);
			return rec.data;
		}
		return null;
	}

	private static class VBOCacheReference extends PhantomReference<Object> {

		private VBOCacheRecord rec;

		private VBOCacheReference(Object referent, VBOCacheRecord rec, ReferenceQueue<Object> q) {
			super(referent, q);
			this.rec = rec;
		}
	}

}
