/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;
import java.util.concurrent.locks.ReentrantLock;

import javax.media.opengl.GL2;
import javax.media.opengl.GLContext;
import javax.media.opengl.glu.GLU;
import javax.vecmath.Matrix4d;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.MediaOptions;
import ch.kuramo.javie.core.MediaSource;
import ch.kuramo.javie.core.MediaOptions.Option;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.QCIntegrationSupport;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.SynchronousTaskThread;
import ch.kuramo.javie.core.services.SynchronousTaskThread.Task;
import ch.kuramo.javie.core.services.SynchronousTaskThread.TaskWithoutResult;

import com.google.inject.Inject;

public class QCCompositionSource implements MediaSource {

	private static final Logger logger = LoggerFactory.getLogger(QCCompositionSource.class);


	private static final Time DEFAULT_DURATION = Time.fromFrameNumber(3600, FrameDuration.FPS_59_94);

	private static final Time FRAME_DURATION = new Time(0, Time.COMMON_TIME_SCALE);

	private static final Size2i DEFAULT_SIZE = new Size2i(640, 480);

	private static final VideoBounds DEFAULT_BOUNDS = new VideoBounds(DEFAULT_SIZE);


	private GLContext glContext;

	private GLU glu;

	private int fboId;

	private long pointer;


	private final SynchronousTaskThread thread;

	private final GLGlobal glGlobal;

	private final RenderContext context;

	private final IVideoRenderSupport vrSupport;

	private final QCIntegrationSupport qciSupport;

	@Inject
	public QCCompositionSource(
			SynchronousTaskThread thread, GLGlobal glGlobal,
			RenderContext context, IVideoRenderSupport vrSupport,
			QCIntegrationSupport qciSupport) {

		super();
		this.thread = thread;
		this.glGlobal = glGlobal;
		this.context = context;
		this.vrSupport = vrSupport;
		this.qciSupport = qciSupport;
	}

	public boolean initialize(final File file) {
		// 拡張子がqtz以外の場合（このメソッドからfalseが返され）その直後に MediaSourceFactoryImpl から dispose() が呼ばれるが、
		// 先にスレッドを開始しておかないと disposeQCRenderer 内の thread.exit() で停止してしまう。
		// FIXME これは SynchronousTaskThreadImpl の方を修正すべきでは。 
		thread.start();

		String fileName = file.getName();
		int lastDot = fileName.indexOf('.');
		if (lastDot == -1 || !"qtz".equals(fileName.substring(lastDot+1).toLowerCase())) {
			thread.exit();
			return false;
		}

		thread.invoke(new TaskWithoutResult() {
			protected void runWithoutResult() throws Exception {
				initQCRenderer(file);
			}
		});
		if (pointer == 0) {
			disposeQCRenderer();
			return false;
		} else {
			return true;
		}
	}

	private void initQCRenderer(File file) {
		ReentrantLock lock = glGlobal.getGlobalLock();
		lock.lock();
		try {
			glContext = glGlobal.createContext();
			glContext.makeCurrent();
		} finally {
			lock.unlock();
		}

		glu = new GLU();

		//glContext.setGL(new DebugGL(glContext.getGL()));
		GL2 gl = glContext.getGL().getGL2();

		int[] fboId_ = new int[1];
		gl.glGenBuffers(1, fboId_, 0);
		fboId = fboId_[0];

		gl.glBindFramebuffer(GL2.GL_FRAMEBUFFER, fboId);


		long pixelFormat = MacOSXOpenGLPixelFormat.createPixelFormat(glContext);
		pointer = initQCRenderer(file.getAbsolutePath(), pixelFormat);
		MacOSXOpenGLPixelFormat.deletePixelFormat(pixelFormat);

		if (pointer != 0) {
			thread.setName(getClass().getSimpleName() + ": " + file.getName());
		}
	}

	public void dispose() {
		disposeQCRenderer();
	}

	private void disposeQCRenderer() {
		thread.exit(new TaskWithoutResult() {
			protected void runWithoutResult() {
				if (pointer != 0) {
					disposeQCRenderer(pointer);
					pointer = 0;
				}

				if (glContext != null) {
					GL2 gl = glContext.getGL().getGL2();

					gl.glBindFramebuffer(GL2.GL_FRAMEBUFFER, 0);
					gl.glDeleteBuffers(1, new int[] { fboId }, 0);
					fboId = 0;

					ReentrantLock lock = glGlobal.getGlobalLock();
					lock.lock();
					try {
						glContext.release();
						glGlobal.destroyContext(glContext);
					} finally {
						lock.unlock();
					}

					glContext = null;
					glu = null;
				}
			}
		});
	}

	public Time getDuration(MediaOptions options) {
							// このMediaSourceでは、DURATIONオプションは常に使用可能。
		if (options == null /*|| !options.isAvailable(Option.DURATION)*/) {
			return DEFAULT_DURATION;
		} else {
			Time duration = options.getDuration();
			return (duration != null) ? duration : DEFAULT_DURATION;
		}
	}

	public Time getVideoFrameDuration(MediaOptions options) {
		return FRAME_DURATION;
	}

	public VideoBounds getVideoFrameBounds(MediaOptions options) {
							// このMediaSourceでは、SIZEオプションは常に使用可能。
		if (options == null /*|| !options.isAvailable(Option.SIZE)*/) {
			return DEFAULT_BOUNDS;
		} else {
			Size2i size = options.getSize();
			return (size == null || size.equals(DEFAULT_SIZE)) ? DEFAULT_BOUNDS : new VideoBounds(size);
		}
	}

	public IVideoBuffer getVideoFrame(Time mediaTime, MediaOptions options) {
		if (pointer == 0) {
			return null;
		}

		VideoBounds bounds = context.getVideoResolution().scale(getVideoFrameBounds(options));
		IVideoBuffer buffer1 = null;
		try {
			buffer1 = vrSupport.createVideoBuffer(bounds);
			buffer1.clear();

			if (mediaTime.timeValue < 0 || !mediaTime.before(getDuration(options))) {
				IVideoBuffer result = buffer1;
				buffer1 = null;
				return result;
			}

			if (renderAtTime(buffer1, mediaTime)) {
				return flipVertical(buffer1);
			} else {
				return null;
			}

		} finally {
			if (buffer1 != null) buffer1.dispose();
		}
	}

	private boolean renderAtTime(IVideoBuffer buffer, final Time mediaTime) {
		final boolean syncWithCamera = qciSupport.isSyncWithCamera();
		final Object[] inputKeyValues = qciSupport.getInputKeyValues();
		qciSupport.clear();

		final double[] prjMatrix, mvMatrix;
		if (syncWithCamera) {
			Camera camera = context.getCamera();
			prjMatrix = camera.getProjection3D();
			mvMatrix = camera.getModelView3D();
		} else {
			prjMatrix = null;
			mvMatrix = null;
		}

		final int texture = buffer.getTexture();
		final int width = buffer.getBounds().width;
		final int height = buffer.getBounds().height;
		final ColorMode colorMode = context.getColorMode();

		// ここまでの処理が完了していないまま thread.invoke() が実行されるとマズいので。
		context.getGL().glFinish();

		return thread.invoke(new Task<Boolean>() {
			public Boolean run() throws Exception {
				GL2 gl = glContext.getGL().getGL2();

				int deptex = 0;
				gl.glPushAttrib(GL2.GL_CURRENT_BIT | GL2.GL_TEXTURE_BIT | GL2.GL_DEPTH_BUFFER_BIT);
				try {
					deptex = createDepthTexture(gl, width, height, colorMode);

					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_2D, texture, 0);
					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_DEPTH_ATTACHMENT, GL2.GL_TEXTURE_2D, deptex, 0);

					gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

					gl.glClearDepth(1.0);
					gl.glClear(GL2.GL_DEPTH_BUFFER_BIT);

					gl.glViewport(0, 0, width, height);

					if (syncWithCamera) {
						double[] qcPrjInvert = calcInvertMatrixOfQuartzComposerProjection(gl, width, height);
						gl.glMatrixMode(GL2.GL_PROJECTION);
						gl.glLoadMatrixd(prjMatrix, 0);
						gl.glMultMatrixd(qcPrjInvert, 0);

						double[] tsr = extractTranslateScaleRotate(mvMatrix);
						gl.glMatrixMode(GL2.GL_MODELVIEW);
						gl.glLoadIdentity();
						gl.glTranslated(tsr[0], -tsr[1], tsr[2]);
						gl.glScaled(tsr[3], tsr[4], tsr[5]);
						gl.glRotated(-Math.toDegrees(tsr[8]), 0, 0, 1);
						gl.glRotated( Math.toDegrees(tsr[7]), 0, 1, 0);
						gl.glRotated(-Math.toDegrees(tsr[6]), 1, 0, 0);

						gl.glTranslated(width*0.5, -height*0.5, 0);
						gl.glScaled(width*0.5, width*0.5, width*0.5); 

					} else {
						gl.glMatrixMode(GL2.GL_PROJECTION);
						gl.glLoadIdentity();
						gl.glMatrixMode(GL2.GL_MODELVIEW);
						gl.glLoadIdentity();
					}

					boolean successful = renderAtTime(pointer, mediaTime.toSecond(), inputKeyValues);
					if (!successful) {
						logger.error("renderAtTime: failed");
					}

					// 処理が完了しないまま、元のスレッドに戻ってしまうとマズいので。
					gl.glFinish();

					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_2D, 0, 0);
					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_DEPTH_ATTACHMENT, GL2.GL_TEXTURE_2D, 0, 0);

					return successful;

				} finally {
					if (deptex != 0) gl.glDeleteTextures(1, new int[] { deptex }, 0);
					gl.glPopAttrib();
				}
			}
		});
	}


	// createDepthTexture は AntiAliasSupportImpl にあるものとほぼ同じ。 

	private static final float[] FLOAT0000 = new float[] { 0, 0, 0, 0 };

	private int createDepthTexture(GL2 gl, int width, int height, ColorMode colorMode) {
		// TODO 常時 GL_DEPTH_COMPONENT32 を指定した方が良いかも？
		int internalFormat = (colorMode == ColorMode.RGBA32_FLOAT)
								? GL2.GL_DEPTH_COMPONENT32 : GL2.GL_DEPTH_COMPONENT;

		int[] texture = new int[1];
		int[] current = new int[1];
		gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 0);
		try {
			gl.glGenTextures(1, texture, 0);
			gl.glBindTexture(GL2.GL_TEXTURE_2D, texture[0]);

			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_NEAREST);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_NEAREST);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_BORDER);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_BORDER);
			gl.glTexParameterfv(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
			gl.glTexImage2D(GL2.GL_TEXTURE_2D, 0, internalFormat,
					Math.max(1, width), Math.max(1, height), 0, GL2.GL_DEPTH_COMPONENT, GL2.GL_FLOAT, null);

			int result = texture[0];
			texture[0] = 0;
			return result;

		} finally {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, current[0]);
			if (texture[0] != 0) gl.glDeleteTextures(1, texture, 0);
		}
	}

	private double[] calcInvertMatrixOfQuartzComposerProjection(GL2 gl, double width, double height) {
		gl.glMatrixMode(GL2.GL_PROJECTION);
		gl.glLoadIdentity();

		double aspect = width/height;
		double fovx = Math.toRadians(60);
		double fovy = 2*Math.atan(Math.tan(fovx/2)/aspect);
		glu.gluPerspective(Math.toDegrees(fovy), aspect, 0.1, 100);
		gl.glTranslated(0, 0, -Math.sqrt(3));	// これはOK。この移動は投影行列の方に乗算される。

		double[] invert = new double[16];
		gl.glGetDoublev(GL2.GL_PROJECTION_MATRIX, invert, 0);

		Matrix4d m = new Matrix4d(invert);
		m.transpose();
		m.invert();

		invert[ 0] = m.m00;
		invert[ 1] = m.m10;
		invert[ 2] = m.m20;
		invert[ 3] = m.m30;
		invert[ 4] = m.m01;
		invert[ 5] = m.m11;
		invert[ 6] = m.m21;
		invert[ 7] = m.m31;
		invert[ 8] = m.m02;
		invert[ 9] = m.m12;
		invert[10] = m.m22;
		invert[11] = m.m32;
		invert[12] = m.m03;
		invert[13] = m.m13;
		invert[14] = m.m23;
		invert[15] = m.m33;

		return invert;
	}

	private double[] extractTranslateScaleRotate(double[] mvMatrix) {
		Matrix4d m = new Matrix4d(mvMatrix);
		m.transpose();
		m.mul(new Matrix4d(1,0,0,0,0,1,0,0,0,0,-1,0,0,0,0,1));

		double tx = m.m03;
		double ty = m.m13;
		double tz = m.m23;
		double sx = Math.sqrt(m.m00*m.m00 + m.m01*m.m01 + m.m02*m.m02);
		double sy = Math.sqrt(m.m10*m.m10 + m.m11*m.m11 + m.m12*m.m12);
		double sz = Math.sqrt(m.m20*m.m20 + m.m21*m.m21 + m.m22*m.m22);
		double rx, ry, rz;
		if (m.m20 == 1) {
			rx = 0;
			ry = -Math.PI / 2;
			rz = -Math.atan2(m.m01, m.m11);
		} else if (m.m20 == -1) {
			rx = 0;
			ry = Math.PI / 2;
			rz = -Math.atan2(m.m01, m.m11);
		} else {
			rx = Math.atan2(m.m21, m.m22);
			ry = Math.asin(-m.m20);
			rz = Math.atan2(m.m10, m.m00);
		}

		return new double[] { tx,ty,tz, sx,sy,sz, rx,ry,rz };
	}

	private IVideoBuffer flipVertical(IVideoBuffer buffer) {
		final VideoBounds bounds = buffer.getBounds();

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();
				GLU glu = context.getGLU();

				gl.glViewport(0, 0, bounds.width, bounds.height);

				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadIdentity();
				glu.gluOrtho2D(0, bounds.width, bounds.height, 0);

				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();

				gl.glColor4f(1, 1, 1, 1);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(0, 0); gl.glVertex2f(0, 0);
				gl.glTexCoord2f(1, 0); gl.glVertex2f(bounds.width, 0);
				gl.glTexCoord2f(1, 1); gl.glVertex2f(bounds.width, bounds.height);
				gl.glTexCoord2f(0, 1); gl.glVertex2f(0, bounds.height);
				gl.glEnd();
			}
		};

		int pushAttribs = GL2.GL_CURRENT_BIT;
		return vrSupport.useFramebuffer(operation, pushAttribs, null, buffer);
	}


	public boolean isVideoAvailable() {
		return true;
	}

	public boolean isAudioAvailable() {
		return false;
	}

	public IAudioBuffer getAudioChunk(Time mediaTime, MediaOptions options) {
		throw new UnsupportedOperationException("audio is not available");
	}

	public MediaOptions validateOptions(MediaOptions options) {
		Option[] availableOptions = {
				Option.DURATION,
				Option.SIZE
		};

		if (options == null) {
			options = new MediaOptions(availableOptions);
		} else {
			options = options.clone();
			options.setAvailableOptions(availableOptions);
		}

		if (options.getDuration() == null) {
			options.setDuration(DEFAULT_DURATION);
		}
		if (options.getSize() == null) {
			options.setSize(DEFAULT_SIZE);
		}

		return options;
	}


	static { System.loadLibrary("QCCompositionSource"); }

	private native long initQCRenderer(String filename, long pixelFormat);

	private native void disposeQCRenderer(long qcRenderer);

	private native boolean renderAtTime(long qcRenderer, double time, Object[] inputKeyValues);

}
