/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.expression;

import java.util.Random;

import noise.ImprovedNoise;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.Function;
import org.mozilla.javascript.Scriptable;

import ch.kuramo.javie.api.Time;

public class RandomNumbers {

	public static final String[] METHOD_NAMES = {
		"seedRandom", "random", "gaussRandom", "noise",
		"wiggle1", "wiggle2", "wiggle3", "wiggle"
	};

	public static Object seedRandom(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		Boolean timeless = Boolean.FALSE;
		Number seed;

		switch (args.length) {
			case 2:
				timeless = (Boolean) Context.jsToJava(args[1], boolean.class);
				// fall through
			case 1:
				seed = (Number) Context.jsToJava(args[0], Number.class);
				break;
			default:
				throw new IllegalArgumentException();
		}

		RandomNumbers rn = (RandomNumbers) cx.getThreadLocal(RandomNumbers.class);
		rn.seedRandom(seed.longValue(), timeless);

		return Context.getUndefinedValue();
	}

	public static Object random(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		return random(cx, args, false);
	}

	public static Object gaussRandom(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		return random(cx, args, true);
	}

	private static Object random(Context cx, Object[] args, boolean gaussian) {
		Object o1 = 0.0;
		Object o2 = 1.0;

		switch (args.length) {
			case 0:
				break;

			case 1:
				o2 = ExpressionUtils.toDoubleOrDoubleArray(args[0]);
				break;

			default:
				o1 = ExpressionUtils.toDoubleOrDoubleArray(args[0]);
				o2 = ExpressionUtils.toDoubleOrDoubleArray(args[1]);
				break;
		}

		RandomNumbers rn = (RandomNumbers) cx.getThreadLocal(RandomNumbers.class);
		Random r = rn.getRandom();

		if (o1 instanceof Double && o2 instanceof Double) {
			double min = (Double) o1;
			double max = (Double) o2;
			return random(r, min, max, gaussian);
		}

		double[] a1 = (o1 instanceof Double) ? new double[] { (Double)o1 } : (double[])o1;
		double[] a2 = (o2 instanceof Double) ? new double[] { (Double)o2 } : (double[])o2;
		double[] a = new double[Math.max(a1.length, a2.length)];

		for (int i = 0; i < a.length; ++i) {
			double min = (i < a1.length) ? a1[i] : 1;	// 入力の配列長が異なるとき、AEの動作は
			double max = (i < a2.length) ? a2[i] : 0;	// なぜかこのように、minの方が1、maxの方が0となっている。
			a[i] = random(r, min, max, gaussian);
		}

		return a;
	}

	private static double random(Random r, double min, double max, boolean gaussian) {
		double d;
		if (gaussian) {
			d = r.nextGaussian() / 3.2894 + 0.5;	// 90%がmin-maxの範囲となる。
		} else {
			d = r.nextFloat();		// nextDouble() だと分布が若干悪い気がする (nextDoubleの実装はnextを2回呼んでいるせい?)
		}
		return d * (max - min) + min;
	}

	public static double noise(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		if (args.length != 1) {
			throw new IllegalArgumentException();
		}

		Object o = ExpressionUtils.toDoubleOrDoubleArray(args[0]);
		double d0, d1, d2;

		if (o instanceof Double) {
			d0 = (Double) o;
			d1 = d2 = 0;
		} else {
			double[] array = (double[]) o;
			d0 = (array.length >= 1) ? array[0] : 0;
			d1 = (array.length >= 2) ? array[1] : 0;
			d2 = (array.length >= 3) ? array[2] : 0;
		}

		// ImprovedNoiseは整数を引数にすると規則的な値を返す場合がある。
		// そのため適当な小数値を足している。
		return ImprovedNoise.noise(
				d0 + 0.2356,
				d1 + 0.6049,
				d2 + 0.7724);
	}

	private static final double[] DOUBLE_0   = new double[] { 0 };
	private static final double[] DOUBLE_00  = new double[] { 0, 0 };
	private static final double[] DOUBLE_000 = new double[] { 0, 0, 0 };

	public static Object wiggle1(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		return wiggle(cx, args, DOUBLE_0)[0];
	}

	public static Object wiggle2(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		return wiggle(cx, args, DOUBLE_00);
	}

	public static Object wiggle3(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		return wiggle(cx, args, DOUBLE_000);
	}

	public static Object wiggle(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		if (args.length < 3 || args.length > 5) {
			throw new IllegalArgumentException();
		}

		Object values = ExpressionUtils.toDoubleOrDoubleArray(args[0]);

		Object[] args2 = new Object[args.length-1];
		System.arraycopy(args, 1, args2, 0, args2.length);

		if (values instanceof Double) {
			return wiggle(cx, args2, new double[] { (Double)values })[0];
		} else {
			return wiggle(cx, args2, (double[])values);
		}
	}

	private static double[] wiggle(Context cx, Object[] args, double[] values) {
		if (args.length < 2 || args.length > 4) {
			throw new IllegalArgumentException();
		}

		double freq = (Double) Context.jsToJava(args[0], double.class);
		double amp = (Double) Context.jsToJava(args[1], double.class);
		int octaves = (args.length >= 3) ? (int) Math.ceil((Double) Context.jsToJava(args[2], double.class)) : 1;
		double ampMult = (args.length >= 4) ? (Double) Context.jsToJava(args[3], double.class) : 0.5;

		RandomNumbers rn = (RandomNumbers) cx.getThreadLocal(RandomNumbers.class);
		return wiggle(freq, amp, octaves, ampMult, rn, rn.time.toSecond(), values);
	}

	private static double[] wiggle(double freq, double amp, int octaves, double ampMult,
							RandomNumbers rn, double time, double[] values) {

		double[] result = new double[values.length];
		for (int i = 0; i < result.length; ++i) {
			result[i] = values[i];
		}

		for (int i = 0; i < octaves; ++i, amp *= ampMult, freq *= 2) {
			double t = time * freq;
			for (int j = 0; j < result.length; ++j) {
				result[j] += amp * ImprovedNoise.noise(
											0.2356 + rn.seed,
											0.6049 + t,
											0.7724 + 73*(j+1));
			}
		}

		return result;
	}

	static double[] wiggle(double freq, double amp, double octaves, double ampMult, double[] values) {
		Context cx = Context.getCurrentContext();
		RandomNumbers rn = (RandomNumbers) cx.getThreadLocal(RandomNumbers.class);
		return wiggle(freq, amp, (int)Math.ceil(octaves), ampMult, rn, rn.time.toSecond(), values);
	}

	static double[] wiggle(double freq, double amp, double octaves, double ampMult, double t, double[] values) {
		Context cx = Context.getCurrentContext();
		RandomNumbers rn = (RandomNumbers) cx.getThreadLocal(RandomNumbers.class);
		return wiggle(freq, amp, (int)Math.ceil(octaves), ampMult, rn, t, values);
	}


	private long seed;

	private boolean timeless;

	private final Time time;

	private Random random;

	public RandomNumbers(long seed, Time time) {
		this.seed = seed;
		this.time = time;
	}

	private Random getRandom() {
		if (random == null) {
			random = new Random(seed + (timeless ? 0 : time.timeValue));
			random.nextFloat();		// 最初の値は分布に偏りがあるようなので捨てる。
		}
		return random;
	}

	private void seedRandom(long seed, boolean timeless) {
		this.seed = seed;
		this.timeless = timeless;
		random = null;
	}

}
